/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import static org.hamcrest.CoreMatchers.*;
import static org.junit.Assert.*;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.util.Arrays;
import java.util.List;

import org.junit.Test;

import com.google.common.collect.Lists;

@SuppressWarnings("unchecked")
public class RSListTest {

  RSList type = new RSList(Types.FLOATING);

  @Test
  public void canCoerceAListOfAnUnderlyingType() throws Exception {
    List<String> listOfStrings = Lists.newArrayList("1.0", "1", "3.4");
    assertThat(
        type.coerce(listOfStrings),
        allOf(
            equalTo(Lists.newArrayList(1.0D, 1.0D, 3.4D)),
            // make sure it's cloning the array - we rely on pipelines and expressions
            // never mutating things and keeping it functional (GL#1434)
            not(sameInstance(listOfStrings))
        )
    );

    assertEquals(type.coerce(new String[] {"1.0", "1", "3.4"}),
        Lists.newArrayList(1.0D, 1.0D, 3.4D));
  }

  @Test
  public void canCoerceAListFromPrimitiveArrays() {

    java.util.List<Double> expectedDoubles = Lists.newArrayList(1.0D, 1.1D, 3.4D);

    assertEquals(type.coerce(new double[] {1.0D, 1.1D, 3.4D}),
        expectedDoubles);

    java.util.List<Double> coercedFloats = (java.util.List<Double>) type.coerce(new float[] {1.0F, 1.1F, 3.4F});
    for (int i = 0; i < coercedFloats.size(); i++) {
      Double value = coercedFloats.get(i);
      assertEquals(expectedDoubles.get(i).doubleValue(), value.doubleValue(), 0.001);
    }

    java.util.List expectedInts = Lists.newArrayList(1D, 2D, 3D);

    long[] sourceLongs = new long[] {1L, 2L, 3L};
    assertEquals(type.coerce(sourceLongs),
        expectedInts);

    // sample one of the array and check it's not been mutated (GL#1434)
    assertThat(sourceLongs[0], is(1L));

    assertEquals(type.coerce(new int[] {1, 2, 3}),
        expectedInts);

    assertEquals(type.coerce(new short[] {1, 2, 3}),
        expectedInts);

    assertEquals(type.coerce(new byte[] {1, 2, 3}),
        expectedInts);
  }

  @Test
  public void canSerializeAListInAndOut() throws Exception {
    java.util.List<String> values = Arrays.asList("foo", "bar", "baz");
    Type listOfStrings = new RSList(Types.TEXT);
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    DataOutputStream dos = new DataOutputStream(bos);
    listOfStrings.toBytes(dos, values);
    assertEquals(values, listOfStrings.fromBytes(new DataInputStream(new ByteArrayInputStream(bos.toByteArray()))));
  }

  @Test
  public void asStructCreatesAStructWithASingleList() throws Exception {
    Type listOfStrings = new RSList(Types.TEXT);
    assertEquals(Struct.of("value", listOfStrings), listOfStrings.asStruct());
  }

  @Test
  public void listsAreEqualIfMemberTypeIsEqual() {
    assertThat(new RSList(Types.INTEGER).equals(new RSList(Types.INTEGER)), is(true));
    Struct nested = Struct.of("name", Types.TEXT, "value", Types.INTEGER);
    assertThat(new RSList(nested).equals(new RSList(nested)), is(true));
  }

  @Test
  public void listsAreNotEqualIfMemberTypesDiffer() {
    assertThat(new RSList(Types.INTEGER).equals(new RSList(Types.TEXT)), is(false));
    Struct nested = Struct.of("name", Types.TEXT, "value", Types.INTEGER);
    assertThat(new RSList(nested).equals(Types.TEXT), is(false));
    assertThat(new RSList(Types.TEXT).equals(new RSList(nested)), is(false));

    Struct nested1 = Struct.of("name", Types.TEXT, "description", Types.TEXT);
    assertThat(new RSList(nested).equals(new RSList(nested1)), is(false));
  }

}
