/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.stream.Collectors;

import org.junit.Test;

import nz.org.riskscape.engine.typeset.TypeSet;
import nz.org.riskscape.engine.typexp.AST.ComplexType;
import nz.org.riskscape.engine.typexp.AST.Constant;
import nz.org.riskscape.engine.typexp.AST.Symbol;
import nz.org.riskscape.engine.typexp.AST.ValueList;
import nz.org.riskscape.engine.typexp.TypeBuilder;
import nz.org.riskscape.engine.typexp.TypeConstructor;


public class LambdaTypeTest {

  TypeBuilder builder = new TypeBuilder() {

    @Override
    public TypeSet getTypeSet() {
      return null;
    }

    @Override
    public Type buildSimpleType(Symbol symbol) {
      return null;
    }

    @Override
    public Type buildComplexType(ComplexType ast) {
      return null;
    }

    @Override
    public Type build(String typeExpression) {
      return null;
    }
  };

  @Test
  public void canBeConstructedFromATypeExpression() throws Exception {
    TypeConstructor constructor = LambdaType.TYPE_INFORMATION.getConstructor();
    LambdaType built;

    built = (LambdaType) constructor.construct(builder, fakeAst());
    assertThat(built.getArgs(), empty());

    built = (LambdaType) constructor.construct(builder, fakeAst("foo"));
    assertThat(built.getArgs(), contains("foo"));

    built = (LambdaType) constructor.construct(builder, fakeAst("foo", "bar"));
    assertThat(built.getArgs(), contains("foo", "bar"));
  }

  @Test
  public void typesAreEqualIfTheyHaveTheSameNumberOfArgs() throws Exception {
    assertThat(LambdaType.NO_ARGS, equalTo(new LambdaType()));

    assertThat(
      new LambdaType("foo"),
      allOf(
        equalTo(new LambdaType("foo")),
        equalTo(new LambdaType("bar")),
        not(equalTo(new LambdaType("foo", "bar")))
      )
    );

    assertThat(
      new LambdaType("foo", "bar"),
      allOf(
        equalTo(new LambdaType("barry", "baz")),
        not(equalTo(new LambdaType("foo")))
      )
    );
  }

  @Test
  public void aScopedVersionCanBeConstructedUsingLambdaTypeAsAStartingPoint() throws Exception {
    ScopedLambdaType scoped = new LambdaType("foo").scoped(Struct.of("bar", Types.TEXT));

    assertThat(scoped.getArgs(), contains("foo"));
    assertThat(scoped.getScopeType(), equalTo(Struct.of("bar", Types.TEXT)));
  }

  @Test
  public void aScopedLambdaTypeIsEqualOnlyWhenArgsAndScopeMatch() throws Exception {
    assertThat(
      new ScopedLambdaType(Struct.EMPTY_STRUCT, "foo"),
      allOf(
        equalTo(new ScopedLambdaType(Struct.EMPTY_STRUCT, "foo")),
        not(equalTo(new ScopedLambdaType(Struct.EMPTY_STRUCT, "bar"))),
        not(equalTo(new ScopedLambdaType(Struct.of("foo", Types.TEXT), "foo")))
      )
    );
  }

  private ComplexType fakeAst(String... idents) {
    return new ComplexType(
      null,
      new ValueList(
        null,
        Arrays.asList(idents).stream().map(str -> new Constant(null, str)).collect(Collectors.toList()))
    );
  }
}
