/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.util.LinkedList;
import java.util.Optional;

import org.geotools.geometry.jts.ReferencedEnvelope;
import org.junit.Test;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

public class GeomTypeTest {

  @Test
  public void detectsGeometryTypes() {
    assertTrue(GeomType.isAllowNull(Types.GEOMETRY));
    assertTrue(GeomType.isAllowNull(Types.LINE));
    assertTrue(GeomType.isAllowNull(Types.POINT));
    assertTrue(GeomType.isAllowNull(Types.POLYGON));

    assertTrue(GeomType.isAllowNull(MultiGeom.MULTI_LINE));
    assertTrue(GeomType.isAllowNull(MultiGeom.MULTI_POINT));
    assertTrue(GeomType.isAllowNull(MultiGeom.MULTI_POLYGON));
  }

  @Test
  public void detectsNullableGeometryTypes() {
    assertTrue(GeomType.isAllowNull(Nullable.GEOMETRY));
    assertTrue(GeomType.isAllowNull(Nullable.of(Types.LINE)));
    assertTrue(GeomType.isAllowNull(Nullable.of(Types.POINT)));
    assertTrue(GeomType.isAllowNull(Nullable.of(Types.POLYGON)));

    assertTrue(GeomType.isAllowNull(Nullable.of(MultiGeom.MULTI_LINE)));
    assertTrue(GeomType.isAllowNull(Nullable.of(MultiGeom.MULTI_POINT)));
    assertTrue(GeomType.isAllowNull(Nullable.of(MultiGeom.MULTI_POLYGON)));
  }

  @Test
  public void detectsReferencedGeometryTypes() {
    CoordinateReferenceSystem crs = mock(CoordinateReferenceSystem.class);
    ReferencedEnvelope envelope = mock(ReferencedEnvelope.class);

    assertTrue(GeomType.isAllowNull(Referenced.of(Types.GEOMETRY, crs, envelope)));
    assertTrue(GeomType.isAllowNull(Nullable.of(Referenced.of(Types.GEOMETRY, crs, envelope))));
  }

  @Test
  public void isReturnsFalseForNonGeometries() {
    assertFalse(GeomType.isAllowNull(Types.TEXT));
    assertFalse(GeomType.isAllowNull(Types.INTEGER));
    assertFalse(GeomType.isAllowNull(Struct.EMPTY_STRUCT));

    assertFalse(GeomType.isAllowNull(Nullable.TEXT));
    assertFalse(GeomType.isAllowNull(Nullable.INTEGER));
    assertFalse(GeomType.isAllowNull(Nullable.of(Struct.EMPTY_STRUCT)));
  }

  @Test
  public void findsSuperTypes() {
    // Point extends GeomType
    assertTrue(Types.POINT.isA(GeomType.class));
    assertThat(Types.POINT.find(GeomType.class), is(Optional.of(Types.POINT)));
  }

  @Test
  public void visitingSeesTheWholeHierarchy() throws Exception {
    assertThat(
      TypeVisitor.bfs(Types.POINT, new LinkedList<Type>(), (list, type) -> list.add(type)),
      contains(Types.POINT, Types.GEOMETRY)
    );

    assertThat(
      TypeVisitor.bfs(Types.POLYGON, new LinkedList<Type>(), (list, type) -> list.add(type)),
      contains(Types.POLYGON, Types.GEOMETRY)
    );

    assertThat(
      TypeVisitor.bfs(Types.LINE, new LinkedList<Type>(), (list, type) -> list.add(type)),
      contains(Types.LINE, Types.GEOMETRY)
    );

    assertThat(
      TypeVisitor.bfs(MultiGeom.MULTI_LINE, new LinkedList<Type>(), (list, type) -> list.add(type)),
      contains(MultiGeom.MULTI_LINE, Types.LINE, Types.GEOMETRY)
    );

    assertThat(
      TypeVisitor.bfs(MultiGeom.MULTI_POINT, new LinkedList<Type>(), (list, type) -> list.add(type)),
      contains(MultiGeom.MULTI_POINT, Types.POINT, Types.GEOMETRY)
    );

    assertThat(
      TypeVisitor.bfs(MultiGeom.MULTI_POLYGON, new LinkedList<Type>(), (list, type) -> list.add(type)),
      contains(MultiGeom.MULTI_POLYGON, Types.POLYGON, Types.GEOMETRY)
    );
  }

}
