/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.util.Arrays;
import java.util.List;

import org.junit.Test;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.PrecisionModel;
import org.locationtech.jts.io.WKTReader;

public class GeomTest {

  public static final String LINE_WKT = "LINESTRING(10 10, 10 30)";

  public static final String POINT_WKT = "POINT(10 10)";

  public static final String POLYGON_WKT = "POLYGON((10 10, 20 10, 15 15, 10 10))";

  private GeometryFactory gf = new GeometryFactory();

  private final Point point = gf.createPoint();
  private final MultiPoint mpoint = gf.createMultiPoint();
  private final LineString line = gf.createLineString();
  private final MultiLineString mline = gf.createMultiLineString();
  private final Polygon polygon = gf.createPolygon();
  private final MultiPolygon mpolygon = gf.createMultiPolygon();

  @Test
  public void testFromJavaType() {
    assertThat(Geom.fromJavaType(LineString.class), is(Types.LINE));
    assertThat(Geom.fromJavaType(MultiLineString.class), is(MultiGeom.MULTI_LINE));

    assertThat(Geom.fromJavaType(Point.class), is(Types.POINT));
    assertThat(Geom.fromJavaType(MultiPoint.class), is(MultiGeom.MULTI_POINT));

    assertThat(Geom.fromJavaType(Polygon.class), is(Types.POLYGON));
    assertThat(Geom.fromJavaType(MultiPolygon.class), is(MultiGeom.MULTI_POLYGON));

    assertThat(Geom.fromJavaType(Geometry.class), is(Types.GEOMETRY));
  }

  @Test
  public void testCoerceLines() {
    assertThat(Types.LINE.coerce(LINE_WKT), instanceOf(LineString.class));
    assertThat(Types.LINE.coerce(line), instanceOf(LineString.class));

    // should be able to coerce geometries of wrong sub type
    coerceFails(Types.LINE, POINT_WKT);
    coerceFails(Types.LINE, point);
    coerceFails(Types.LINE, mpoint);
    coerceFails(Types.LINE, mline);
    coerceFails(Types.LINE, polygon);
    coerceFails(Types.LINE, mpolygon);
  }

  @Test
  public void testCoercePoints() {
    assertThat(Types.POINT.coerce(POINT_WKT), instanceOf(Point.class));
    assertThat(Types.POINT.coerce(point), instanceOf(Point.class));

    // should be able to coerce geometries of wrong sub type
    coerceFails(Types.POINT, LINE_WKT);
    coerceFails(Types.POINT, mpoint);
    coerceFails(Types.POINT, line);
    coerceFails(Types.POINT, mline);
    coerceFails(Types.POINT, polygon);
    coerceFails(Types.POINT, mpolygon);
  }

  @Test
  public void testCoercePolygon() {
    assertThat(Types.POLYGON.coerce(POLYGON_WKT), instanceOf(Polygon.class));
    assertThat(Types.POLYGON.coerce(polygon), instanceOf(Polygon.class));

    // should be able to coerce geometries of wrong sub type
    coerceFails(Types.POLYGON, LINE_WKT);
    coerceFails(Types.POLYGON, mpoint);
    coerceFails(Types.POLYGON, line);
    coerceFails(Types.POLYGON, mline);
    coerceFails(Types.POLYGON, mpolygon);
  }

  @Test
  public void canReadAndWriteSomeGeometryAsBytes() throws Exception {
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    DataOutputStream dos = new DataOutputStream(bos);
    gf = new GeometryFactory(new PrecisionModel(), 420);
    WKTReader reader = new WKTReader(gf);
    Geom geom = Types.GEOMETRY;

    List<Geometry> geoms = Arrays.asList(
      reader.read(POINT_WKT), reader.read(LINE_WKT), reader.read(POLYGON_WKT)
    );

    for (Geometry geometry : geoms) {
      geom.toBytes(dos, geometry);
      dos.flush();
      byte[] writtenBytes = bos.toByteArray();
      Object read = geom.fromBytes(new DataInputStream(new ByteArrayInputStream(writtenBytes)));
      assertThat(read, equalTo(geometry));
      Geometry readGeometry = (Geometry) read;
      // should have the right srid, too (don't think Geometry#equals considers this)
      assertEquals(gf.getSRID(), readGeometry.getSRID());
      bos.reset();
    }
  }

  void coerceFails(Type type, Object toCoerce) {
    try {
      type.coerce(toCoerce);
    } catch (CoercionException e) {
      return;
    }
    fail("Should have raise a CoercionException but didn't");
  }

}
