/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.plugin;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import lombok.Data;

public class ExtensionPointsTest {

  ExtensionPoints subject = new ExtensionPoints();

  @Data
  static class FooFeature implements PluginFeature {
    private final String foo;
  }

  interface OtherFeature extends PluginFeature {
    String value();
  }

  @Data
  static class BarFeature implements PluginFeature, OtherFeature {
    private final String bars;

    @Override
    public String value() {
      return "other " + bars;
    }
  }

  @Data
  static class BazFeature implements PluginFeature {
    private final String barry;
  }


  @Test
  public void canRegisterSomeFeaturesToVariousExtensionPoints() {
    subject.addExtensionPoint(FooFeature.class);
    subject.addExtensionPoint(BarFeature.class);
    subject.addExtensionPoint(BazFeature.class);

    subject.addFeature(new FooFeature("some foo"));
    subject.addFeature(new BazFeature("barry island"));
    subject.addFeature(new FooFeature("more foo"));

    assertThat(
      subject.getFeaturesOfType(FooFeature.class),
      contains(new FooFeature("some foo"), new FooFeature("more foo"))
    );

    assertThat(
      subject.getFeaturesOfType(BazFeature.class),
      contains(new BazFeature("barry island"))
    );
  }

  @Test
  public void aFeatureCanBelongToManyExtensionPoints() throws Exception {
    subject.addExtensionPoint(BarFeature.class);
    subject.addExtensionPoint(OtherFeature.class);

    subject.addFeature(new BarFeature("some bar"));

    assertThat(
      subject.getFeaturesOfType(BarFeature.class),
      contains(new BarFeature("some bar"))
    );

    assertThat(
      subject.getFeaturesOfType(OtherFeature.class),
      contains(new BarFeature("some bar"))
    );
  }

  @Test
  public void aFeatureIsIgnoredIfItDoesNotBelongToAnyExtensions() throws Exception {
    subject.addExtensionPoint(BarFeature.class);

    assertThat(subject.addFeature(new BarFeature("some bar")), equalTo(true));
    assertThat(subject.addFeature(new FooFeature("no more foo")), equalTo(false));

    assertThat(
      subject.getFeaturesOfType(BarFeature.class),
      contains(new BarFeature("some bar"))
    );

    // we added a bar feature, which is also otherFeature, but we didn't register the other feature, so comes back empty
    assertThat(
      subject.getFeaturesOfType(OtherFeature.class),
      empty()
    );

    // foo feature doesn't even figure
    assertThat(
      subject.getFeaturesOfType(FooFeature.class),
      empty()
    );
  }

}
