/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import org.junit.Test;

import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.problem.Problems;

public class RealizedPipelineTest {

  RealizedStep fooStep = RealizedStep.named("foo");
  RealizedStep barStep = RealizedStep.named("bar").withDependencies(fooStep);
  RealizedStep bazStep = RealizedStep.named("baz").withDependencies(barStep);

  ExecutionContext context = mock(ExecutionContext.class);
  RealizedPipeline empty = RealizedPipeline.empty(context, PipelineDeclaration.EMPTY);

  @Test
  public void canBuildAPipelineByAddingNewSteps() throws Exception {
    RealizedPipeline built = empty.add(fooStep).add(barStep);
    // pipelines should be immutable
    assertThat(built, not(sameInstance(empty)));

    // test what the constructed pipeline contains
    assertThat(built.getStep("foo").get(), sameInstance(fooStep));
    assertThat(built.getStep("bar").get(), sameInstance(barStep));
    assertThat(built.getStep("baz").isPresent(), is(false));

    assertThat(built.getDependents(fooStep), contains(barStep));
    assertThat(built.getDependents(barStep), empty());
  }

  @Test(expected = IllegalArgumentException.class)
  public void addingAStepFailsIfNameIsNotUnique() throws Exception {
    // throws IllegalArgumentException
    empty.add(fooStep).add(RealizedStep.named("foo"));
  }

  @Test(expected = IllegalArgumentException.class)
  public void addingAStepFailsIfItHasMissingDependencies() throws Exception {
    // throws IllegalArgumentException - bar depends on foo
    empty.add(barStep);
  }

  @Test
  public void canGetFailures() throws Exception {
    fooStep = fooStep.withProblems(Problems.foundWith("us"));
    barStep = barStep.withDependencies(fooStep).withProblems(Problems.foundWith("step1"));

    RealizedPipeline realizedPipeline = empty.add(fooStep).add(barStep).addProblems(Problems.foundWith("you"));

    assertThat(
      realizedPipeline.getFailures(),
      containsInAnyOrder(
        // note that foo's problem is wrapped with a cannotRealize error that refers to the ast for context
        PipelineProblems.get().cannotRealize(fooStep.getAst()).withChildren(Problems.foundWith("us")),
        Problems.foundWith("you")
        // bar's problem is ignored, because it depends on foo, which failed
      )
    );
  }
}
