/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

import java.util.Arrays;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Types;

public class NullSafeFunctionTest {

  RiskscapeFunction target = mock(RiskscapeFunction.class);
  RiskscapeFunction wrapped;

  @Before
  public void setup() {
    when(target.getArgumentTypes()).thenReturn(Arrays.asList(Types.INTEGER, Types.BOOLEAN));
    when(target.getReturnType()).thenReturn(Types.FLOATING);
    when(target.call(any())).thenReturn(1D);

    wrapped = NullSafeFunction.wrap(target);
  }


  @Test
  public void correctlyAdjustsTypes() throws Exception {
    assertEquals(Arrays.asList(Nullable.INTEGER, Nullable.BOOLEAN), wrapped.getArgumentTypes());
    assertEquals(Nullable.FLOATING, wrapped.getReturnType());
  }

  @Test
  public void callsTheTargetFunctionIfArgsAreSatisfied() throws Exception {
    assertEquals(1D, wrapped.call(Arrays.asList(1L, true)));
  }

  @Test
  public void shortcircuitsTheTargetFunctionIfAnyArgsAreNull() throws Exception {
    assertNull(wrapped.call(Arrays.asList(null, true)));
    assertNull(wrapped.call(Arrays.asList(1L, null)));
    assertNull(wrapped.call(Arrays.asList(null, null)));
  }

  @Test
  public void doesNotShortCircuitIfOriginalArgWasDeclaredNullable() throws Exception {
    when(target.getArgumentTypes()).thenReturn(Arrays.asList(Nullable.INTEGER, Types.BOOLEAN));
    wrapped = NullSafeFunction.wrap(target);

    assertTrue(wrapped.getReturnType().isNullable());
    assertEquals(1D, wrapped.call(Arrays.asList(1L, true)));
    assertEquals(1D, wrapped.call(Arrays.asList(null, true)));
    assertEquals(null, wrapped.call(Arrays.asList(2L, null)));
    assertEquals(null, wrapped.call(Arrays.asList(null, null)));
  }

  @Test
  public void doesNotWrapAnotherNullSafeFunction() {
    assertSame(wrapped, NullSafeFunction.wrap(wrapped));

    when(target.getArgumentTypes()).thenReturn(Arrays.asList(Nullable.INTEGER, Nullable.BOOLEAN));
    wrapped = NullSafeFunction.wrap(target);
    assertSame(wrapped, target);

    assertEquals(1D, wrapped.call(Arrays.asList(null, null)));
  }

}
