/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import static nz.org.riskscape.engine.CoreMatchers.*;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.Matchers.*;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.junit.Assert.*;

import java.net.URI;
import java.util.List;

import org.junit.Before;
import org.junit.Test;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.typeset.TypeSet;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.ResultOrProblems;

public class JavaFunctionTest {

  Struct returnType = Struct.of("name", Types.TEXT, "value", Types.INTEGER);
  Struct delegateReturnType = returnType.add("dr", Types.FLOATING);
  TypeSet typeSet;
  // currently Java functions shouldn't need to use this
  RealizationContext realizationContext = null;

  @Before
  public void setup() {
    typeSet = new TypeSet();
    typeSet.add("return-type", returnType);
  }

  @Test
  public void validationCausesNoProblemsFromDefaultDeleteValidate() {
    JavaFunction function = JavaFunction.withId("to-test")
        .withReturnType(returnType)
        .calling((args) -> {
          return Tuple.of(delegateReturnType, "name", "delegated", "value", 1000L).set("dr", 0.5d);
        });
    ResultOrProblems<Boolean> valid = function.validate(realizationContext);
    assertThat(valid.get(), is(true));
    assertThat(valid.getProblems(), is(empty()));
  }

  @Test
  public void validationWillFailOnBadId() {
    JavaFunction.Delegate delegate = (List<Object> arguments) -> "delegated";
    JavaFunction function = JavaFunction.withIdAndSource("bad id", URI.create("test-file.txt"))
        .withReturnType(Types.TEXT)
        .calling(delegate);

    ResultOrProblems<Boolean> valid = function.validate(realizationContext);
    assertThat(valid.hasErrors(), is(true));

    assertThat(valid.getProblems(), contains(
            isProblem(Severity.ERROR, is(
                "ID 'bad id' is not allowed. ID's may only contain letters, digits, ':', '-' and '_' characters"))
    ));
  }

  @Test
  public void validationWillWrapProblemsFromDelegateValidate() {
    JavaFunction.Delegate delegate = new JavaFunction.Delegate() {
      @Override
      public Object apply(List<Object> arguments) {
        return "delegated";
      }

      @Override
      public List<Problem> validate() {
        return Lists.newArrayList(
            Problem.error("mock error"),
            Problem.fatal(null, "mock fatal")
        );
      }
    };
    JavaFunction function = JavaFunction.withIdAndSource("to-test", URI.create("test-file.txt"))
        .withReturnType(Types.TEXT)
        .calling(delegate);

    ResultOrProblems<Boolean> valid = function.validate(realizationContext);
    assertThat(valid.hasErrors(), is(true));

    assertThat(valid.getProblems(), contains(
            isProblem(Severity.ERROR, is("mock error")),
            isProblem(Severity.FATAL, is("mock fatal"))
    ));
  }

}
