/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

import com.google.common.collect.ImmutableMap;

import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

public class BoundParametersTest {

  Parameter required = Parameter.required("foo", Long.class);
  Parameter twoOrThree = Parameter.range("twoOrThree", Long.class, 2, 3);
  Parameter bar = Parameter.required("hasDefault", Long.class, -1L);
  ParameterSet parameterSet = ParameterSet.from(required, twoOrThree, bar);

  BindingContext bindingContext = mock(BindingContext.class);

  @Before
  public void setup() {
    when(bindingContext.bind(any(), Mockito.any(Parameter.class))).thenAnswer(inv -> {
      String unbound = inv.getArgument(0);
      try {
        return ResultOrProblems.of(Long.valueOf(unbound));
      } catch (NumberFormatException e) {
        return ResultOrProblems.failed(Problems.foundWith(unbound));
      }
    });
  }

  @Test
  public void canMergeTwoIncompletesToMakeAWhole() {
    BoundParameters one = parameterSet.bind(bindingContext, ImmutableMap.of("foo", Arrays.asList("1")));
    BoundParameters two = parameterSet.bind(bindingContext, ImmutableMap.of("twoOrThree", Arrays.asList("1", "3")));

    BoundParameters merged = one.merge(two);
    assertTrue(merged.isValid());
    assertThat(merged.getValueMap(), allOf(
        hasEntry("foo", Arrays.asList(1L)),
        hasEntry("twoOrThree", Arrays.asList(1L, 3L)),
        hasEntry("hasDefault", Arrays.asList(-1L)) // keeps the default
    ));
  }

  @Test
  public void problemIfMergedParametersNotComplete() {
    BoundParameters one = parameterSet.bind(bindingContext, ImmutableMap.of("foo", Arrays.asList("1")));
    BoundParameters two = parameterSet.bind(bindingContext, ImmutableMap.of("twoOrThree", Arrays.asList("1")));

    BoundParameters merged = one.merge(two);
    assertFalse(merged.isValid());
    assertThat(merged.getValidationProblems(), contains(
        ParamProblems.get().wrongNumberGiven(twoOrThree.getName(), twoOrThree.getArity(), 1)
    ));
    assertThat(merged.getValueMap(), allOf(
        hasEntry("foo", Arrays.asList(1L)),
        hasEntry("twoOrThree", Arrays.asList(1L)),
        hasEntry("hasDefault", Arrays.asList(-1L)) // keeps the default
    ));
  }

  @Test
  public void parametersAreReplaced() {
    BoundParameters one = parameterSet.bind(bindingContext, ImmutableMap.of("foo", Arrays.asList("1")));
    BoundParameters two = parameterSet.bind(bindingContext, ImmutableMap.of(
        "foo", Arrays.asList("33"),
        "twoOrThree", Arrays.asList("1", "3")));

    BoundParameters merged = one.merge(two);
    assertTrue(merged.isValid());
    assertThat(merged.getValueMap(), allOf(
        hasEntry("foo", Arrays.asList(33L)), //this guy got replaced
        hasEntry("twoOrThree", Arrays.asList(1L, 3L)),
        hasEntry("hasDefault", Arrays.asList(-1L)) // keeps the default
    ));
  }

  @Test
  public void defaultValuesDonotReplaceRealValues() {
    BoundParameters one = parameterSet.bind(bindingContext, ImmutableMap.of(
        "foo", Arrays.asList("1"),
        "twoOrThree", Arrays.asList("1", "3"),
        "hasDefault", Arrays.asList("10")
    ));
    BoundParameters two = parameterSet.bind(bindingContext, ImmutableMap.of("foo", Arrays.asList("10")));
    BoundParameters merged = one.merge(two);
    assertTrue(merged.isValid());
    assertThat(merged.getValueMap(), allOf(
        hasEntry("foo", Arrays.asList(10L)),
        hasEntry("twoOrThree", Arrays.asList(1L, 3L)),
        hasEntry("hasDefault", Arrays.asList(10L))  // keeps value from one
    ));
  }

  @Test
  public void extraneousParametersAreNotLost() {
    BoundParameters one = parameterSet.bind(bindingContext, ImmutableMap.of(
        "foo", Arrays.asList("1"),
        "bar", Arrays.asList("barbar")));
    BoundParameters two = parameterSet.bind(bindingContext, ImmutableMap.of(
        "twoOrThree", Arrays.asList("1", "3"),
        "baz", Arrays.asList("bazza")));

    BoundParameters merged = one.merge(two);
    assertTrue(merged.isValid());
    assertThat(merged.getValueMap(), allOf(
        hasEntry("foo", Arrays.asList(1L)),
        hasEntry("twoOrThree", Arrays.asList(1L, 3L)),
        hasEntry("hasDefault", Arrays.asList(-1L)) // keeps the default
    ));

    assertThat(merged.getExtraneous(), allOf(
        hasEntry("bar", Arrays.asList("barbar")),
        hasEntry("baz", Arrays.asList("bazza"))
    ));
  }

  @Test
  public void bindErrorsAreStillAProblem() {
    BoundParameters one = parameterSet.bind(bindingContext, ImmutableMap.of("foo", Arrays.asList("1")));
    BoundParameters two = parameterSet.bind(bindingContext, ImmutableMap.of("twoOrThree", Arrays.asList("ten")));

    BoundParameters merged = one.merge(two);
    assertFalse(merged.isValid());
    assertThat(merged.getValidationProblems(), contains(
        Problems.foundWith("ten") // this is what our mock binding context does
    ));
    assertThat(merged.getValueMap(), allOf(
        hasEntry("foo", Arrays.asList(1L))
    ));
  }

  @Test(expected = IllegalArgumentException.class)
  public void cannotMergeIfContextDifferent() {
    BindingContext otherBindingContext = mock(BindingContext.class);
    when(otherBindingContext.bind(any(), Mockito.any(Parameter.class))).thenAnswer(inv -> {
      String unbound = inv.getArgument(0);
      try {
        return ResultOrProblems.of(Long.valueOf(unbound));
      } catch (NumberFormatException e) {
        return ResultOrProblems.failed(Problems.foundWith(unbound));
      }
    });

    BoundParameters one = parameterSet.bind(bindingContext, ImmutableMap.of("foo", Arrays.asList("1")));
    BoundParameters two = parameterSet.bind(otherBindingContext, ImmutableMap.of("twoOrThree", Arrays.asList("ten")));

    one.merge(two);
  }

  @Test(expected = IllegalArgumentException.class)
  public void cannotMergeIfParameterSetsDifferent() {
    ParameterSet otherParameterSet = ParameterSet.from(required, twoOrThree);
    BoundParameters one = parameterSet.bind(bindingContext, ImmutableMap.of("foo", Arrays.asList("1")));
    BoundParameters two = otherParameterSet.bind(bindingContext, ImmutableMap.of("twoOrThree", Arrays.asList("ten")));

    one.merge(two);
  }

  private BoundParameters seed(Map<String, List<?>> valueMap, Map<String, List<?>> unbound) {
    return new BoundParameters(parameterSet, bindingContext, valueMap, unbound, Collections.emptyList());
  }

}
