/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine;

import static org.hamcrest.Matchers.*;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeMatcher;
import org.hamcrest.core.IsAnything;
import org.hamcrest.core.IsSame;

import nz.org.riskscape.engine.i18n.MessageKey;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problem.Severity;

public class CoreMatchers {

  public static Matcher<Problem> isProblem(Severity severity, Matcher<String> messageMatcher) {
    return isProblem(
        new IsSame<Severity>(severity),
        messageMatcher,
        IsAnything.anything());
  }

  public static Matcher<Problem> isProblem(
      Severity severity,
      Matcher<String> messageMatcher,
      Matcher<?> subMatchers
  ) {
    return isProblem(
        new IsSame<Severity>(severity),
        messageMatcher,
        subMatchers);
  }

  public static Matcher<Problem> isProblem(
      Matcher<Severity> severity,
      Matcher<String> messageMatcher,
      Matcher<?> subMatchers
  ) {
    return new TypeSafeMatcher<Problem>() {

      @Override
      public void describeTo(Description description) {
        description
            .appendText("Problem [")
            .appendDescriptionOf(severity)
            .appendDescriptionOf(messageMatcher)
            .appendDescriptionOf(subMatchers)
            .appendText("]");
      }

      @Override
      protected boolean matchesSafely(Problem item) {
        return severity.matches(item.getSeverity())
            && messageMatcher.matches(item.getMessage())
            && subMatchers.matches(item.getChildren());
      }

      @Override
      protected void describeMismatchSafely(Problem item, Description mismatchDescription) {
        if (!severity.matches(item.getSeverity())) {
          mismatchDescription.appendText("wrong severity ").appendValue(item.getSeverity());
        } else if (!messageMatcher.matches(item.getMessage())) {
          mismatchDescription.appendText("wrong message: ");
          messageMatcher.describeMismatch(item.getMessage(), mismatchDescription);
        } else {
          mismatchDescription.appendText("wrong children: ");
          subMatchers.describeMismatch(item.getChildren(), mismatchDescription);
        }
      }
    };
  }

  public static Matcher<MessageKey> isMessage(Matcher<?> codeMatcher, Matcher<?> defaultMessageMatcher) {
    return isMessage(codeMatcher, defaultMessageMatcher, is(emptyArray()));
  }

  public static Matcher<MessageKey> isMessage(
      Matcher<?> codeMatcher,
      Matcher<?> defaultMessageMatcher,
      Matcher<?> argumentsMatcher) {
    return new TypeSafeMatcher<MessageKey>() {

      @Override
      public void describeTo(Description description) {
        description
            .appendText("MessageKey [")
            .appendDescriptionOf(codeMatcher)
            .appendDescriptionOf(defaultMessageMatcher)
            .appendDescriptionOf(argumentsMatcher)
            .appendText("]");
      }

      @Override
      protected boolean matchesSafely(MessageKey item) {
        return codeMatcher.matches(item.getCode())
            && defaultMessageMatcher.matches(item.getDefaultMessage())
            && argumentsMatcher.matches(item.getMessageArguments());
      }

      @Override
      protected void describeMismatchSafely(MessageKey item, Description mismatchDescription) {
        if (!codeMatcher.matches(item.getCode())) {
          mismatchDescription.appendText("wrong code ").appendValue(item.getCode());
        } else if (!defaultMessageMatcher.matches(item.getDefaultMessage())) {
          mismatchDescription.appendText("wrong default message: ");
          defaultMessageMatcher.describeMismatch(item.getDefaultMessage(), mismatchDescription);
        } else {
          mismatchDescription.appendText("wrong arguments: ");
          argumentsMatcher.describeMismatch(item.getMessageArguments(), mismatchDescription);
        }
      }
    };
  }

}
