/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.dsl;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeMatcher;
import org.junit.Test;

import lombok.Getter;
import nz.org.riskscape.dsl.Lexer.Tokens;

/**
 * The lexer suports tracking indents for whitespace sensitive DSLs.  We used it in the classifier functions , but those
 * are mostly dead now.  We used to cover off this feature in the LexerTest, but it's no longer compatible with the
 * expression languages tokens (it doesn't count tabs as their own tokens and that's a requisite for this feature).
 */
public class LexerIndentingTest {

  enum TestToken implements TokenType {
    WORD(LexingMatcher.forPattern("^[a-z]+"), false),
    SPACE(LexingMatcher.forChar(' '), true),
    TAB(LexingMatcher.forChar('\t'), true),
    NEWLINE(LexingMatcher.forChar('\n'), true),
    EOF(LexingMatcher.forPattern("^$"), true);

    TestToken(LexingMatcher matcher, boolean whitespace) {
      this.matcher = matcher;
      this.whitespace = whitespace;
    }

    final LexingMatcher matcher;
    @Getter
    final boolean whitespace;

    @Override
    public LexingMatcher matcher() {
      return matcher;
    }
  }

  Lexer<TestToken> lexer;

  public void lex(String source) {
    Tokens<TestToken> tokens = new Tokens<>(TestToken.class, TestToken.EOF, List.of(TestToken.NEWLINE));
    // using rl tokens, but that's just to avoid having to define a set of tokens just for these tests
    lexer = new Lexer<>(tokens, source);
  }

  @Test
  public void tokensCanCompareIndenting() throws Exception {
    lex(""
        + " foo\n"
        + "  bar\n"
    );

    Token foo = lexer.next();
    Token bar = lexer.next();

    assertTrue(bar.moreIndented(foo));
    assertFalse(foo.moreIndented(bar));
  }

  @Test
  public void mixedIndentingThrowsAnError() throws Exception {
    lex(""
        + "\tfoo\n"
        + "  bar\n"
    );

    Token foo = lexer.next();
    Token bar = lexer.next();

    try {
      foo.moreIndented(bar);
      fail();
    } catch (InconsistentIndentException e) {

    }
  }

  @Test
  public void rewindingCorrectlyResetsTheNumberOfIndentsScannedOnTheirLine() throws Exception {
    lex(""
        + "foo\n"
        + "  bar\n"
        + "\tbaz\n"
        + "\t  more foo \n"
    );

    assertThat(lexer.next(), allOf(hasValue("foo"), hasIndents()));
    Token bar = lexer.next();
    assertThat(bar, allOf(hasValue("bar"), hasIndents(" ", " ")));
    assertThat(lexer.next(), allOf(hasValue("baz"), hasIndents("\t")));
    assertThat(lexer.next(), allOf(hasValue("more"), hasIndents("\t", " ",  " ")));
    assertThat(lexer.next(), allOf(hasValue("foo"), hasIndents("\t", " ", " ")));

    lexer.rewind(bar);

    assertThat(lexer.next(), allOf(hasValue("bar"), hasIndents(" ", " ")));
    assertThat(lexer.next(), allOf(hasValue("baz"), hasIndents("\t")));
    assertThat(lexer.next(), allOf(hasValue("more"), hasIndents("\t", " ",  " ")));
    assertThat(lexer.next(), allOf(hasValue("foo"), hasIndents("\t", " ", " ")));

    // seek to the indent before the bar token
    lexer.rewind(bar.getPrevious());

    assertThat(lexer.next(), allOf(hasValue("bar"), hasIndents(" ", " ")));
    assertThat(lexer.next(), allOf(hasValue("baz"), hasIndents("\t")));
    assertThat(lexer.next(), allOf(hasValue("more"), hasIndents("\t", " ",  " ")));
    assertThat(lexer.next(), allOf(hasValue("foo"), hasIndents("\t", " ", " ")));
  }

  @Test
  public void tokensCanRecordTheNumberOfIndentsScannedOnTheirLine() throws Exception {
    lex(""
        + "foo\n"
        + "  bar\n"
        + "\tbaz\n"
        + "\t  more foo \n"
    );

    assertThat(lexer.next(), allOf(hasValue("foo"), hasIndents()));
    assertThat(lexer.next(), allOf(hasValue("bar"), hasIndents(" ", " ")));
    assertThat(lexer.next(), allOf(hasValue("baz"), hasIndents("\t")));
    assertThat(lexer.next(), allOf(hasValue("more"), hasIndents("\t", " ",  " ")));
    assertThat(lexer.next(), allOf(hasValue("foo"), hasIndents("\t", " ", " ")));
  }

  private Matcher<Token> hasIndents(String... indentsRest) {
    List<String> indents = Arrays.asList(indentsRest);
    return new TypeSafeMatcher<Token>(Token.class) {

      @Override
      public void describeTo(Description description) {
        description.appendText("token with indents ").appendValue(indents);
      }

      @Override
      protected void describeMismatchSafely(Token item, Description mismatchDescription) {
        mismatchDescription.appendText("token ").appendValue(item).appendText(" with indents ")
            .appendValue(item.getIndents());
      }

      @Override
      protected boolean matchesSafely(Token item) {
        Iterator<String> toMatch = indents.iterator();
        for (Token indent : item.getIndents()) {
          if (!toMatch.hasNext() || !toMatch.next().equals(indent.value)) {
            return false;
          }
        }

        if (toMatch.hasNext()) {
          return false;
        }
        return true;
      }
    };
  }
  private Matcher<Token> hasValue(String value) {
    return new TypeSafeMatcher<Token>(Token.class) {

      @Override
      public void describeTo(Description description) {
        description.appendText("token with value ").appendValue(value);
      }

      @Override
      protected void describeMismatchSafely(Token item, Description mismatchDescription) {
        mismatchDescription.appendText("was ").appendValue(item);
      }

      @Override
      protected boolean matchesSafely(Token item) {
        return item.value.equals(value);
      }
    };
  }

}
