/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape;

import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.ReflectionUtils.SearchOptions;
import nz.org.riskscape.engine.RiskscapeException;

public class ReflectionUtilsTest {

  public abstract static class Base implements Comparable<Base> {

  }

  public static class A extends Base {
    @Override
    public int compareTo(Base o) {
      return 0;
    }
  }
  public static class AA extends A {
  }

  public static class B extends Base {

    @Override
    public int compareTo(Base o) {
      return 0;
    }
  }


  @Test
  public void testAncestorLookup() throws Exception {
    assertEquals(A.class, ReflectionUtils.findCommonAncestorOfType(A.class, A.class, A.class).get());
    assertFalse(ReflectionUtils.findCommonAncestorOfType(String.class, A.class, A.class).isPresent());
    assertEquals(Object.class, ReflectionUtils.findCommonAncestorOfType(String.class, A.class, Object.class).get());
    assertEquals(Base.class, ReflectionUtils.findCommonAncestorOfType(A.class, B.class, Base.class).get());
    assertEquals(Base.class, ReflectionUtils.findCommonAncestorOfType(A.class, B.class, Base.class).get());
    assertEquals(Base.class, ReflectionUtils.findCommonAncestorOfType(A.class, AA.class, Base.class).get());

    assertEquals(Base.class, ReflectionUtils.findCommonAncestorOfType(A.class, A.class, Comparable.class).get());
    assertEquals(Base.class, ReflectionUtils.findCommonAncestorOfType(A.class, B.class, Comparable.class).get());
    assertEquals(Base.class, ReflectionUtils.findCommonAncestorOfType(A.class, B.class, Comparable.class).get());
    assertEquals(Base.class, ReflectionUtils.findCommonAncestorOfType(A.class, AA.class, Comparable.class).get());
  }

  public interface CoolAPI {

  }

  public interface ExperimentalService extends CoolAPI {

  }

  public class DirectlyImplemented implements CoolAPI {

  }

  public class Extended extends DirectlyImplemented {

  }

  public class OnInterface implements ExperimentalService {

  }

  public class Lots extends Extended implements CoolAPI {

  }

  @Test
  public void implementedInterfaceLookup() throws Exception {
    assertSame(
        DirectlyImplemented.class,
        ReflectionUtils.findImplementingClass(CoolAPI.class, DirectlyImplemented.class));

    assertSame(
        DirectlyImplemented.class,
        ReflectionUtils.findImplementingClass(CoolAPI.class, Extended.class));

    assertSame(
        ExperimentalService.class,
        ReflectionUtils.findImplementingClass(CoolAPI.class, OnInterface.class, SearchOptions.SEARCH_INTERFACES));
  }

  @Test(expected = RiskscapeException.class)
  public void willNotSearchInterfacesUnlessAsked() throws Exception {
    ReflectionUtils.findImplementingClass(CoolAPI.class, OnInterface.class);
  }

  @Test(expected = RiskscapeException.class)
  public void implementsSearchFailsIfSameInterfaceImplementedTwice() throws Exception {
    ReflectionUtils.findImplementingClass(CoolAPI.class, Lots.class);
  }

}
