/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import lombok.NonNull;

public class ListUtils {

  /**
   * Fluid, immutable list concatenation method that concatenates two lists, returning a new list with the contents of
   * list1 and list2 appended in order.
   */
  @SuppressWarnings("unchecked")
  public static <T> List<T> concat(List<? extends T> list1, List<? extends T> list2) {
    if (list1.isEmpty()) {
      return (List<T>) list2;
    }
    if (list2.isEmpty()) {
      return (List<T>) list1;
    }

    List<T> newList = new ArrayList<>(list1.size() + list2.size());
    newList.addAll(list1);
    newList.addAll(list2);
    return newList;
  }

  /**
   * Fluid, immutable list appending method that appends an element at the end of a given list, returning a new list
   */
  public static <T> List<T> append(List<? extends T> list, T element) {
    if (list.isEmpty()) {
      return Collections.singletonList(element);
    }

    List<T> newList = new ArrayList<>(list.size() + 1);
    newList.addAll(list);
    newList.add(element);
    return newList;
  }

  /**
   * Fluid, immutable list prepending method that prepends the given element to the given list, returning a new list
   */
  public static <T> List<T> prepend(T element, List<? extends T> list) {
    if (list.isEmpty()) {
      return Collections.singletonList(element);
    }

    List<T> newList = new ArrayList<>(list.size() + 1);
    newList.add(element);
    newList.addAll(list);
    return newList;
  }


  /**
   * Fluid, immutable list prepending method that prepends the given element to the given list, returning a new list
   */
  @SafeVarargs
  public static <T> List<T> prepend(@NonNull T element, T... rest) {
    if (rest.length == 0) {
      return Collections.singletonList(element);
    }

    List<T> newList = new ArrayList<>(rest.length + 1);
    newList.add(element);
    for (int i = 0; i < rest.length; i++) {
      newList.add(rest[i]);
    }
    return newList;
  }
}
