/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.rl.ast;

import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.rl.TokenTypes;

@RequiredArgsConstructor @EqualsAndHashCode(callSuper = false) @Getter @Setter
public class StructDeclaration extends BaseExpr {

  /**
   * @return an anonymous attribute declaration wrapping the given expression.
   */
  public static Member anonMember(Expression expression) {
    return new Member(null, expression, null);
  }

  /**
   * Returns a json style (foo: bar) struct member
   */
  public static Member jsonStyleMember(Token ident, Expression expression) {
    return new Member(ident, expression, null);
  }

  /**
   * Returns a json style (foo: bar) struct member
   */
  public static Member jsonStyleMember(String ident, Expression expression) {
    return new Member(TokenTypes.identToken(ident), expression, null);
  }

  /**
   * Returns an sql style (bar as foo) struct member
   */
  public static Member sqlStyleMember(Token ident, Expression expression, Token as) {
    return new Member(ident, expression, as);
  }


  /**
   * Returns a select-all ({*}) struct member
   */
  public static Member selectAllMember(Token asterisk) {
    return new Member(asterisk, new SelectAllExpression(asterisk), null);
  }

  @Data()
  public static class Member {

    @Nullable
    private final Token identifier;
    private final Expression expression;

    /**
     * `true` if the member is a vanilla select-all operator, e.g. `{foo, *, bar}`
     */
    private final boolean selectAll;

    /**
     * `true` if the member has a trailing select-all operator, e.g. `{foo, bar.*, baz}`
     */
    private final boolean selectAllOnReceiver;
    @Nullable
    private final Token as; // null when json style

    private Member(Token ident, Expression expression, Token asToken) {
      this.identifier = ident;
      this.expression = expression;
      this.selectAll = expression instanceof SelectAllExpression
          && identifier != null && identifier.type == TokenTypes.MULTIPLY;

      this.selectAllOnReceiver = expression.isA(PropertyAccess.class)
          .map(PropertyAccess::isTrailingSelectAll)
          .orElse(false);

      this.as = asToken;
    }

    /**
     * @return the declared name of this attribute, or empty if the name was left off
     */
    public Optional<String> getName() {
      return identifier == null ? Optional.empty() : Optional.of(identifier.getValue());
    }

    /**
     * @deprecated use getMemberName instead - the AST is no longer responsible for supplying a default
     * member name to use when the user doesn't supply one
     */
    @Deprecated
    public String getAttributeName() {
      if (identifier != null) {
        return identifier.getValue();
      }
      // Dots in the produced attribute names would be a little misleading (they look like nested prop access)
      // so we replace any dots with underscore. This is only done when the attribute name is inferred from
      // the source expression. If the user has asked for the dots, that's what they get.
      return expression.toSource().replaceAll("\\.", "_");
    }

    public void appendSource(StringBuilder appendTo) {
      if (isJSONStyle()) {
        appendTo.append(identifier.rawValue()).append(": ");
      }

      if (isSelectAll()) {
        appendTo.append(identifier.rawValue());
      } else {
        wrap(expression).appendSource(appendTo);
      }

      if (isSQLStyle()) {
        appendTo.append(" as ").append(identifier.rawValue());
      }
    }

    /**
     * @return an AttrDeclaration of the same type and the same identifier but a new expression.
     */
    public Member cloneWithExpression(Expression newExpression) {
      return new Member(getIdentifier(), newExpression, this.as);
    }

    public String toSource() {
      StringBuilder sb = new StringBuilder();
      appendSource(sb);
      return sb.toString();
    }

    /**
     * @return true if this member has no identifier, i.e. the name should be inferred during realization
     */
    public boolean isAnonymous() {
      return identifier == null;
    }

    /**
     * @return true if the member's name was declared with an as, e.g. `foo as bar`
     */
    public boolean isSQLStyle() {
      return as != null;
    }

    /**
     * @return true if the member's name was declared with a colon, e.g. `bar: foo`
     */
    public boolean isJSONStyle() {
      return !isAnonymous() && !isSelectAll() && !isSQLStyle();
    }
  }

  @Getter
  private final List<Member> members;

  @Getter
  private final Optional<Pair<Token, Token>> boundary;

  @Override
  protected void appendSource(StringBuilder appendTo) {
    join(appendTo, members, (sb, m) -> m.appendSource(sb), ", ", "{", "}");
  }

  @Override
  public <T, R> R accept(ExpressionVisitor<T, R> visitor, T data) {
    return visitor.visit(this, data);
  }

  /**
   * Shortcut for `getMembers().get(index)`
   * @return the AttrDeclaration at the given zero-based index
   * @throws IndexOutOfBoundsException if the index is < 0 or > size
   */
  public Member getMember(int index) {
    return getMembers().get(index);
  }

  public StructDeclaration withNewMembers(List<Member> newMembers, Optional<Pair<Token, Token>> newBoundary) {
    return new StructDeclaration(newMembers, newBoundary);
  }

}
