/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.rl.ast;

import java.util.Optional;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.engine.util.Pair;

/**
 * A 'select all' expression is an asterisk by itself (i.e. not indexing some other expression), and is currently only
 * supported to appear within a struct expression, e.g.
 * `{*, foo} or {foo, *}.`
 *
 * A select all expression will add any members from the input scope to the struct you are declaring, so something
 * like... `input(expression: {foo: 1}) -> select({*, bar: 2})` would yield `{foo: 1, bar: 2}`
 *
 * A related expression is a {@link PropertyAccess} expression with a trailing select all, e.g. `foo().bar.*`
 */
@RequiredArgsConstructor @EqualsAndHashCode(callSuper = false) @Getter @Setter
public class SelectAllExpression extends BaseExpr {

  private final Token token;

  @Override
  protected void appendSource(StringBuilder appendTo) {
   appendTo.append("*");
  }

  @Override
  public Optional<Pair<Token, Token>> getBoundary() {
    return Optional.of(Pair.of(token, token));
  }

  @Override
  public <T, R> R accept(ExpressionVisitor<T, R> visitor, T data) {
    return visitor.visit(this, data);
  }

}
