/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.rl.ast;

/**
 * Subclass of {@link ExpressionVisitor} that does a traversal through the AST, without modifying anything.   Useful
 * for visitors that only want to inspect a specific part of an expression.

 * @param <T> return type and data type
 */
public class MinimalVisitor<T> implements ExpressionVisitor<T, T> {

  @Override
  public T visit(BinaryOperation expression, T data) {
    expression.getLhs().accept(this, data);
    expression.getRhs().accept(this, data);
    return data;
  }

  @Override
  public T visit(BracketedExpression expression, T data) {
    expression.getExpression().accept(this, data);
    return data;
  }

  @Override
  public T visit(Constant expression, T data) {
    return data;
  }

  @Override
  public T visit(FunctionCall expression, T data) {
    expression.getArguments().forEach(arg -> arg.getExpression().accept(this, data));
    return data;
  }

  @Override
  public T visit(ListDeclaration expression, T data) {
    expression.getElements().forEach(el -> el.accept(this, data));
    return data;
  }

  @Override
  public T visit(PropertyAccess expression, T data) {
    expression.getReceiver().ifPresent(rcvr -> rcvr.accept(this, data));
    return data;
  }

  @Override
  public T visit(StructDeclaration expression, T data) {
    expression.getMembers().forEach(m -> m.getExpression().accept(this, data));
    return data;
  }

  @Override
  public T visit(SelectAllExpression expression, T data) {
    return data;
  }

  @Override
  public T visit(Lambda expression, T data) {
    expression.getExpression().accept(this, data);
    return data;
  }

  @Override
  public T visit(ParameterToken parameterToken, T data) {
    return data;
  }

}
