/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.rl.ast;

import java.util.List;
import java.util.Optional;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.rl.TokenTypes;

/**
 * A lambda expression in the RiskScape language is a self-contained expression that can be applied to a list of
 * arguments.  A lambda expression can be in the form: `( [ARG1[, ARGn]] ) -> EXPRESSION` or `ARG -> EXPRESSION` for a
 * one-arg lambda.
 *
 * A classic example for the use of a lambda function is for applying a mapping operation to a list - this can be
 * expressed with a lambda expression like so: `map([1, 2, 3, 4], element -> element * 2)`.
 */
@RequiredArgsConstructor @EqualsAndHashCode(callSuper = false) @Getter @Setter
public class Lambda extends BaseExpr {

  @Getter
  private final Token left;

  /**
   * Names of the arguments as given in the source
   */
  @Getter
  private final List<Token> arguments;

  /**
   * The expression to be evaluated, scoped to the list of arguments
   */
  @Getter
  private final Expression expression;

  @Override
  public Optional<Pair<Token, Token>> getBoundary() {
    return Optional.of(Pair.of(left, expression.getBoundary().get().getRight()));
  }

  @Override
  public <T, R> R accept(ExpressionVisitor<T, R> visitor, T data) {
    return visitor.visit(this, data);
  }

  @Override
  protected void appendSource(StringBuilder appendTo) {
    // if it came in without brackets, it goes out without brackets
    boolean useBrackets = left.type != TokenTypes.IDENTIFIER;
    join(appendTo, arguments, (sb, a) -> sb.append(a.rawValue()), ", ", useBrackets ? "(" : "", useBrackets ? ")" : "");

    appendTo.append(" -> ");
    wrap(expression).appendSource(appendTo);
  }


}
