/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.rl.ast;

import java.util.List;

import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.problem.ProblemPlaceholder;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.StandardCodes;

/**
 * Factory for creating problems involving an expression
 */
public interface ExpressionProblems extends ProblemFactory {

  /**
   * Handy accessor to save on precious keystrokes
   */
  static ExpressionProblems get() {
    return Problems.get(ExpressionProblems.class);
  }

  /**
   * General purpose error for invalid expressions that cannot be parsed.
   * Produces an error: "Failed to parse [expectedThing] expression '[expression]'
   * e.g. Failed to parse type expression 'foo.bar'
   */
  static Problem cannotParse(Object expectedThing, String expression) {
    ProblemPlaceholder exprContext = ProblemPlaceholder.of(Expression.class, expression);
    return Problem.error(StandardCodes.CANT_PARSE, expectedThing, exprContext).affecting(exprContext);
  }

  /**
   * Similar to {@link #cannotParse(Object, String)} but for cases where
   * mentioning 'parsing' is less appropriate. Produces an error:
   *  "'[expression]' is not a valid [typeOfThing] expression"
   */
  Problem notAValid(Class<?> typeOfThing, String expression);

  /**
   * Something required a constant expression, but was given a non-constant
   * expression, e.g. "Expected expression 'foo + bar' to be a constant value"
   */
  Problem constantRequired(Expression expr);

  /**
   * expr is expected to be of requiredType but is not.
   */
  Problem mismatch(Expression expr, Class<? extends Expression> requiredType, String example);

  Problem noSuchOperatorFunction(String op, List<Type> argTypes);

  /**
   * The user has done something like `{foo: bar.*}` which is probably a mistake - it should be either
   * `{bar.*}` or `{foo: bar}`
   */
  Problem pointlessSelectAllInStruct(StructDeclaration.Member memberDeclaration);

  /**
   * The user has done something like `bar.*` outside of a struct, which makes no sense
   */
  Problem pointlessSelectAll(PropertyAccess expression);

  /**
   * A second * has been put in a struct
   */
  Problem duplicateSelectAll(Token duplicate);

  /**
   * A parent problem that wraps a more specific problem within the expression
   */
  Problem malformedExpression(String expression);

  /**
   * Given during realization when a .* is found against a non-struct type, like a list or a number
   */
  Problem selectAllRequiresAStruct(StructDeclaration.Member memberDeclaration, Type resultType);

  /**
   * Give the user some idea of what to do when an unexpected token exception occurs
   */
  Problem malformedExpressionTip();

  /**
   * Wraps other problems to give the context for realization failing
   */
  Problem failedToRealize(Expression expression, Type inputType);

  /**
   * @return from function realization when a lambda expression has the wrong number of args
   */
  Problem lambdaArityError(Expression givenExpression, int found, int expected);

  /**
   * returned when an aggregate expression has property access expressions that aren't within an aggregate function
   */
  Problem propertyOutsideOfAggregationFunction(PropertyAccess expression);

  Problem aggregateFunctionNotSupportedInExpression(FunctionCall fc);

  /**
   * @return when a property expression fails to realize because a member doesn't exist
   */
  Problem noSuchStructMember(String path, List<String> members);

  /**
   * Generic error that says that an expression is no good in the current context as parameters are not allowed, i.e.
   * there's no opportunity for the user to replace them, so it's never going to work.
   */
  Problem parametersNotAllowed(ParameterToken found);

  Problem canNotReplaceMember(String name, StructDeclaration.Member first, StructDeclaration.Member second);

  /**
   * Special case parsing error for when the user gives us an empty string
   */
  Problem emptyStringNotValid();

}
