/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.rl.ast;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import nz.org.riskscape.rl.ast.StructDeclaration.Member;

/**
 * A {@link ExpressionVisitor} for converting expressions in some way.
 *
 * For simple expressions this implementation will return the simple expression.
 *
 * For complex expressions a new instance of the expression is returned with converted sub expressions.
 *
 * By default, any tokens associated with the AST will no longer share the same source - if they need to be in
 * agreement, then the tokens can be 'normalized' by re-parsing the expression
 *
 * @param <T> data passed around to assist with conversion
 */
public class ExpressionConverter<T> implements ExpressionVisitor<T, Expression> {

  @Override
  public Expression visit(BinaryOperation expression, T data) {
    return new BinaryOperation(
        expression.getLhs().accept(this, data),
        expression.getOperator(),
        expression.getRhs().accept(this, data));
  }

  @Override
  public Expression visit(BracketedExpression expression, T data) {
    return new BracketedExpression(expression.getExpression().accept(this, data), expression.getBoundary());
  }

  @Override
  public Expression visit(Constant expression, T data) {
    return expression;
  }

  @Override
  public Expression visit(FunctionCall expression, T data) {
    List<FunctionCall.Argument> convertedArgs = expression.getArguments().stream()
        .map(a -> new FunctionCall.Argument(a.getExpression().accept(this, data), a.getNameToken()))
        .collect(Collectors.toList());

    return new FunctionCall(expression.getIdentifier(), convertedArgs, expression.getBoundary());
  }

  @Override
  public Expression visit(ListDeclaration expression, T data) {
    List<Expression> converted = expression.getElements().stream()
        .map(e -> e.accept(this, data))
        .collect(Collectors.toList());
    return new ListDeclaration(converted, expression.getBoundary());
  }

  @Override
  public Expression visit(PropertyAccess expression, T data) {
    return new PropertyAccess(
      expression.getReceiver().map(r -> r.accept(this, data)),
      expression.getIdentifiers()
    );
  }

  @Override
  public Expression visit(StructDeclaration expression, T data) {
    List<Member> mappedMembers = new ArrayList<>();
    for (Member member : expression.getMembers()) {
      mappedMembers.add(member.cloneWithExpression(member.getExpression().accept(this, data)));
    }
    return expression.withNewMembers(mappedMembers, expression.getBoundary());
  }

  @Override
  public Expression visit(SelectAllExpression expression, T data) {
    return expression;
  }

  @Override
  public Expression visit(Lambda expression, T data) {
    return new Lambda(
        expression.getLeft(),
        expression.getArguments(),
        expression.getExpression().accept(this, data)
    );
  }

  @Override
  public Expression visit(ParameterToken parameterToken, T data) {
    return parameterToken;
  }

}
