/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.rl.ast;

import static nz.org.riskscape.rl.TokenTypes.*;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.rl.TokenTypes;

@RequiredArgsConstructor @EqualsAndHashCode(callSuper = false) @Getter @Setter
public class BinaryOperation extends BaseExpr {

  public static final Map<TokenTypes, TokenTypes> NORMALIZED_TOKENS = ImmutableMap.of(
      TokenTypes.KEYWORD_AND, TokenTypes.AND,
      TokenTypes.KEYWORD_OR, TokenTypes.OR
  );

  public static final List<EnumSet<?>> PRIORITY = ImmutableList.of(
      EnumSet.of(POW),
      EnumSet.of(DIVIDE),
      EnumSet.of(MULTIPLY),
      EnumSet.of(PLUS),
      EnumSet.of(MINUS),
      EnumSet.of(EQUALS, NOT_EQUALS, LESS_THAN_EQUAL, GREATER_THAN_EQUAL, LESS_THAN, GREATER_THAN),
      EnumSet.of(AND, KEYWORD_AND, OR, KEYWORD_OR) //logical comparisions are after numeric so "x > y and a = 3" works
  );

  private final Expression lhs;
  private final Token operator;
  private final Expression rhs;

  @Override
  protected void appendSource(StringBuilder appendTo) {
    wrap(lhs).appendSource(appendTo);
    appendTo.append(" ").append(operator.rawValue()).append(" ");
    wrap(rhs).appendSource(appendTo);
  }

  /**
   * @return a 'normalized' {@link TokenTypes} for the given operator - it gives the non-keyword version of the
   * operator, e.g. it will return OR instead of KEYWORD_OR
   */
  public TokenTypes getNormalizedOperator() {
    TokenTypes tt = (TokenTypes) operator.type;
    return NORMALIZED_TOKENS.getOrDefault(tt, tt);
  }

  @Override
  public <T, R> R accept(ExpressionVisitor<T, R> visitor, T data) {
    return visitor.visit(this, data);
  }

  @Override
  public Optional<Pair<Token, Token>> getBoundary() {
    if (getLhs().getBoundary().isPresent() && getRhs().getBoundary().isPresent()) {
      return Optional.of(Pair.of(getLhs().getBoundary().get().getLeft(), getRhs().getBoundary().get().getRight()));
    } else {
      return Optional.empty();
    }
  }

}
