/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.rl;

import java.util.Set;

import lombok.Getter;
import nz.org.riskscape.dsl.ParseException;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.dsl.TokenType;
import nz.org.riskscape.dsl.UnexpectedTokenException;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.rl.ast.ExpressionProblems;

/**
 * Thrown by a ExpressionParser when we got an unexpected token, indicating that
 * the expression must be malformed.
 * Note that the Token in this case does not have any absolute line/column info,
 * so doesn't hold much relevance to the user. I.e. the column only relates to the
 * expression string itself, not where the expression appears in the overall file.
 */
@SuppressWarnings("serial")
public class MalformedExpressionException extends ParseException {

  private static Problem constructProblem(String expression, UnexpectedTokenException ex) {
    Problem parent = ExpressionProblems.get().malformedExpression(expression);
    // if it's a standard unexpected token problem, show a tip
    if (ex.hasDefaultProblem()) {
      Problem tip = ExpressionProblems.get().malformedExpressionTip();
      return parent.withChildren(ex.getProblem(), tip);
    } else {
      return parent.withChildren(ex.getProblem());
    }

  }

  public MalformedExpressionException(String expression, UnexpectedTokenException exception) {
    super(constructProblem(expression, exception));
    this.expression = expression;
    this.exception = exception;
  }

  @Getter
  private UnexpectedTokenException exception;

  @Getter
  private final String expression;

  @Override
  public synchronized Throwable getCause() {
    return exception;
  }

  public Token getGot() {
    return exception.getGot();
  }

  public Set<? extends TokenType> getExpected() {
    return exception.getExpected();
  }
}
