/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.problem;

import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import nz.org.riskscape.problem.Problem.Severity;

/**
 * A list of {@link Problem}s.  Package private as it APIs do not need to be exposed to it directly
 */
class ProblemList implements Problems {

  static ProblemList of() {
    return new ProblemList(List.of());
  }

  static ProblemList copyOf(List<Problem> children) {
    assert children.size() > 1 : "Should not be creating lists from less than 2 problems";
    return new ProblemList(List.copyOf(children));
  }

  private final List<Problem> list;

  private ProblemList(List<Problem> wrapping) {
    // NB List.copyOf is smart enough to do a noop if the list is already an immutable list
    this.list = List.copyOf(wrapping);
  }

  @Override
  public boolean isEmpty() {
    return list.isEmpty();
  }

  @Override
  public Iterator<Problem> iterator() {
    return list.iterator();
  }

  @Override
  public Optional<Problem> toProblem() {
    if (isEmpty()) {
      return Optional.empty();
    } else {
      return Optional.of(Problems.toSingleProblem(list));
    }
  }

  @Override
  public Stream<Problem> stream() {
    return list.stream();
  }

  @Override
  public List<Problem> toList() {
    return list;
  }

  @Override
  public Severity getSeverity() {
    return list.stream().map(p -> p.severity).max((s1, s2) -> s1.compareTo(s2)).orElse(Severity.INFO);
  }
}
