/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.pipeline.ast;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NonNull;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.engine.pipeline.Step;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.rl.TokenTypes;
import nz.org.riskscape.rl.ast.FunctionCall;
import nz.org.riskscape.rl.ast.FunctionCall.Argument;

/**
 * Defines a step. May optionally include parameter defaults and name for the step.
 */
@EqualsAndHashCode(callSuper = false)
public class StepDefinition extends StepDeclaration {

  @Getter
  private final FunctionCall step;

  @Getter @NonNull
  private final Optional<Token> nameToken;

  public StepDefinition(String id) {
    this(
        new FunctionCall(TokenTypes.identToken(id), Collections.emptyList()),
        Optional.empty(),
        Optional.empty()
    );
  }

  public StepDefinition(FunctionCall fc, Optional<Token> nameToken, Optional<Token> namedInputToken) {
    super(fc.getIdentifier(), namedInputToken);
    this.step = fc;
    this.nameToken = nameToken;
  }

  /**
   * @return an identifier used to link the step definition to a {@link Step} that the engine provides for pipeline
   * processing
   */
  public String getStepId() {
    return identToken.getValue();
  }

  /**
   * @return a name that the user assigned to this step definition, or empty if the step is anonymous
   */
  public Optional<String> getName() {
    return nameToken.map(Token::getValue);
  }


  public List<Argument> getStepParameters() {
    return step.getArguments();
  }

  /**
   * @return the ast token for the identifier used to link the step definition to a {@link Step} that the engine
   * provides for pipeline processing
   */
  public Token getStepIdToken() {
    return step.getIdentifier();
  }

  @Override
  public Optional<Pair<Token, Token>> getBoundary() {
    return step.getBoundary().map(stepBoundary -> {
      return Pair.of(stepBoundary.getLeft(), nameToken.orElse(stepBoundary.getRight()));
    });
  }

  @Override
  protected void appendSource(StringBuilder appendTo) {
    appendTo.append(step.toSource());

    if (nameToken.isPresent()) {
      appendTo.append(" as ").append(nameToken.get().rawValue());
    }
  }

  @Override
  protected void appendString(StringBuilder appendTo) {
    if (!nameToken.isPresent()) {
      // we use a ? to make it clear that it has no alias
      appendTo.append("?");
    }

    appendTo.append(getName().orElse(getIdent())).append("(");
    if (!step.getArguments().isEmpty()) {
      appendTo.append("…");
    }
    appendTo.append(")");

    if (namedInputToken.isPresent()) {
      appendTo.append(".").append(namedInputToken.get().getValue());
    }
  }
}
