/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.pipeline.ast;

import java.util.Optional;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.dsl.Token;

/**
 * AST for a single step.
 */
@RequiredArgsConstructor @EqualsAndHashCode(callSuper = false, doNotUseGetters = true)
public abstract class StepDeclaration extends BaseExpr {

  /**
   * The first identifier that starts this step declaration.  Will be either the name of the referenced step
   * (StepReference) or the
   * id of a step being defined (StepDefinition).
   */
  @Getter
  protected final Token identToken;

  /**
   * The identifier that specifies a named input which accepts the previous step in the chain's output, e.g. the baz in
   * `foo -> bar.baz`.  Valid for references and definitions.
   */
  @Getter
  protected final Optional<Token> namedInputToken;

  /**
   * Get the input name, if any, that the output should be directed to. Only applied to {@link StepReference}s.
   *
   * @return input name, if any
   */
  public Optional<String> getNamedInput() {
    return namedInputToken.map(t -> t.getValue());
  }

  /**
   * @return the string value of this declaration's identifier
   */
  public String getIdent() {
    return identToken.getValue();
  }

  /**
   * Functional cast for a step, useful for definition or reference specific functional code in favour of
   * if-for-instance style logic
   */
  public <T extends StepDeclaration> Optional<T> isA(Class<T> test) {
    return test.isInstance(this) ? Optional.of(test.cast(this)) : Optional.empty();
  }

}
