/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.pipeline;

import java.util.function.Function;

import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.pipeline.ast.StepDeclaration;
import nz.org.riskscape.pipeline.ast.StepDefinition;

/**
 * Slightly-over-engineered way of determining an implicit name, if possible, for a step definition.
 *
 * Putting this logic in its own 'unit' makes it a little easier for us to reuse this logic across modules.  I don't
 * expect this API needs to last - once we've changed our pipeline APIs to go from AST->RealizedPipeline in one call, we
 * can probably drop this interface
 */
@FunctionalInterface
public interface StepNamingPolicy {

  /**
   * @return a function that broadens the argument type of the given function so that it will return the name of any
   * step declaration, not just a definition.
   */
  static Function<StepDeclaration, String> broaden(Function<StepDefinition, String> stepNameFunction) {
    return sd -> {
      return sd.isA(StepDefinition.class).map(s -> stepNameFunction.apply(s)).orElse(sd.getIdent());
    };
  }

  /**
   * @return a function that gives the name for a step definition, returning a unique generated name if none was
   * assigned.  Should return explicitly assigned names, too, to make name lookup logic simpler.  The function will be
   * particular to the given `ast` - do not reuse it for different ones!
   */
  Function<StepDefinition, String> getStepNameFunction(PipelineDeclaration ast);


}
