/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.pipeline;

import static nz.org.riskscape.dsl.ConditionalParse.*;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

import nz.org.riskscape.dsl.Lexer;
import nz.org.riskscape.dsl.LexerException;
import nz.org.riskscape.dsl.ParseException;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.engine.RiskscapeIOException;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.pipeline.ast.StepChain;
import nz.org.riskscape.pipeline.ast.StepDeclaration;
import nz.org.riskscape.pipeline.ast.StepDefinition;
import nz.org.riskscape.pipeline.ast.StepReference;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.TokenTypes;
import nz.org.riskscape.rl.ast.FunctionCall;

public class PipelineParser {

  public static final PipelineParser INSTANCE = new PipelineParser();

  /**
   * Convenience function to parse a parameterized pipeline from the given Resource
   */
  public static ResultOrProblems<PipelineDeclaration> parseParameterizedPipeline(Resource source) {
    return parsePipeline(source, true);
  }

  /**
   * Convenience function to parse a pipeline (no parameters allowed) from the given Resource
   */
  public static ResultOrProblems<PipelineDeclaration> parsePipeline(Resource source) {
    return parsePipeline(source, false);
  }

  private static ResultOrProblems<PipelineDeclaration> parsePipeline(Resource source, boolean allowParameters) {
    try {
      return ResultOrProblems.of(
          INSTANCE.parsePipeline(source.getContentAsString(), allowParameters)
              // assign the location
              .withMetadata(PipelineMetadata.ANONYMOUS.withLocation(source.getLocation()))
      );
    } catch (RiskscapeIOException | ParseException | LexerException e) {
      return ResultOrProblems.failed(Problems.caught(e));
    }
  }

  /**
  * Parses a pipeline from the DSL
  *
  * @param pipelineSource
  * @return pipeline declaration
  */
  public PipelineDeclaration parsePipeline(String pipelineSource) {
    return parsePipeline(pipelineSource, false);
  }

  /**
   * Parses a pipeline from the DSL, allowing $parameters to appear in the sauce
   *
   * @param pipelineSource
   * @return pipeline declaration
   */
   public PipelineDeclaration parsePipelineAllowParameters(String pipelineSource) {
     return parsePipeline(pipelineSource, true);
   }


  private PipelineDeclaration parsePipeline(String pipelineSource, boolean allowParameters) {
   Lexer<TokenTypes> lexer = new Lexer<>(TokenTypes.tokens(), pipelineSource);

//   PipelineDeclaration pipeline = new PipelineDeclaration();
   List<StepChain> chains = new LinkedList<>();
   //Indicates that the last step should be chained to the current step
   List<StepDeclaration> stepChain = null;
   List<Token> links = null;
   while (lexer.peekType() != TokenTypes.EOF) {
     // at this point as are expecting
     // - step declaration
     // - property access (in case of chaining to named input)
     StepDeclaration currentStep = parseStep(lexer, allowParameters);
     if (stepChain == null) {
       // step chains are created on demand, so we don't end up with empty ones.
       stepChain = new LinkedList<>();
       links = new LinkedList<>();
     }
     stepChain.add(currentStep);

     Optional<Token> chaining = lexer.consumeIf(TokenTypes.CHAIN);

     if (!chaining.isPresent()) {
       // this chain is complete
       // add it to the pipeline
       chains.add(new StepChain(stepChain, links));
       stepChain = null;
     } else {
       links.add(chaining.get());
     }
   }
   if (stepChain != null) {
     // If we have got to this point and stepChain is not null then the it's a syntax error as
     // we are expecting the next pipeline definition. So we call lexer expect with the expected tokens
     // to generate the error
     lexer.expect(TokenTypes.IDENTIFIER, TokenTypes.QUOTED_IDENTIFIER);
   }
   lexer.expect(TokenTypes.EOF);
   return new PipelineDeclaration(chains);
 }

 /**
  * Parses a step from the pipeline DSL.
 * @param allowParameters
  *
  * @return step declaration
  */
 StepDeclaration parseStep(Lexer<TokenTypes> lexer, boolean allowParameters) {
   // at this point as are expecting
   // - function call
   // - step reference
   StepDeclaration parsed = lexer.tryThese(
       parseIf("function-expression", Arrays.asList(
           ExpressionParser.IDENTIFIERS, EnumSet.of(TokenTypes.LPAREN)
       ), () -> parseStepDefinition(lexer, allowParameters)),
       parseIf("property-access", Arrays.asList(
           ExpressionParser.IDENTIFIERS
       ), () -> parseStepReference(lexer))
   );
   return parsed;
 }

 private StepReference parseStepReference(Lexer<TokenTypes> lexer) {
     Token stepName = lexer.expect(TokenTypes.IDENTIFIER, TokenTypes.QUOTED_IDENTIFIER);
     Optional<Token> namedInput;
     if (lexer.peekType() == TokenTypes.INDEX) {
       lexer.next();
       namedInput = Optional.of(lexer.expect(TokenTypes.IDENTIFIER, TokenTypes.QUOTED_IDENTIFIER));
     } else {
       namedInput = Optional.empty();
   }

   return new StepReference(stepName, namedInput);
 }

 StepDefinition parseStepDefinition(Lexer<TokenTypes> lexer, boolean allowParameters) {

   FunctionCall fc = ExpressionParser.INSTANCE.parseFunctionExpression(lexer);

   if (!allowParameters) {
     ExpressionParser.INSTANCE.checkForParameters(fc);
   }

   // if there's a dot after the function call, then we parse the identifier to use as the step's name
   Optional<Token> inputName = lexer.consumeIf(TokenTypes.INDEX)
       .map(as -> lexer.expect(TokenTypes.IDENTIFIER, TokenTypes.QUOTED_IDENTIFIER));

   // a step definition my be named, otherwise fall back to using the step id as a name
   Optional<Token> name = lexer.consumeIf(TokenTypes.KEYWORD_AS)
       .map(as -> lexer.expect(TokenTypes.IDENTIFIER, TokenTypes.QUOTED_IDENTIFIER));

   return new StepDefinition(fc, name, inputName);
 }
}
