/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.ini;

import java.util.List;
import java.util.function.Predicate;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.dsl.Token;

/**
 * AST for an .ini file section, comprising the header (`[section]`) and {@link KeyValue}s beneath it
 */
@RequiredArgsConstructor
public class Section {

  /**
   * The `[section]` part.  The value of this token is the content within the brackets.
   */
  private final Token header;

  @Getter
  private final List<KeyValue> members;

  /**
   * The name of the section, i.e. the content within the square brackets with leading and trailing whitespace removed.
   */
  public String getName() {
    return header.getValue().trim();
  }

  @Override
  public String toString() {
    return "Section[" + getName() + "]";
  }

  /**
   * @return all the values in this section for a given key, removing blank values.  Never returns null, instead will
   * return an empty list if the key is not defined.
   */
  public List<String> getAll(String key) {
    return members.stream()
        .filter(kv -> kv.getKey().equals(key))
        .map(KeyValue::getValue)
        .filter(Predicate.not(String::isBlank))
        .toList();
  }

  public Token getHeaderToken() {
    return header;
  }
}
