/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.ini;

import lombok.Data;
import nz.org.riskscape.dsl.Token;

/**
 * AST for A single key-value defined in an .ini file, e.g. `foo = bar`
 */
@Data
public class KeyValue {

  private final Token key;
  private final Token value;

  /**
   * @return the key part, i.e. the bit before the equals sign.
   */
  public String getKey() {
    return key.getValue();
  }

  /**
   * @return the AST of the key
   */
  public Token getKeyToken() {
    return key;
  }

  /**
   * @return the defined value.  Might be a blank string, but leading and trailing whitespace should be removed.
   */
  public String getValue() {
    if (value.getType().equals(IniTokens.MULTILINE_VALUE)) {
      // Multi-line values are designed to retain whitespace within the quotes".
      return value.getValue();
    }
    // Whereas single line values should be trimmed to remove any trailing or leading whitespace around
    // the value (See #1548)
    return value.getValue().trim();
  }

  /**
   * @return the AST of the value
   */
  public Token getValueToken() {
    return value;
  }

  /**
   * @return true if the value is just whitespace or empty.
   */
  public boolean isBlank() {
    return getValue().isBlank();
  }
}
