/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.ini;

import java.util.List;

import lombok.Getter;
import nz.org.riskscape.dsl.Lexer.Tokens;
import nz.org.riskscape.dsl.LexingMatcher;
import nz.org.riskscape.dsl.TokenType;

/**
 * Tokens and matchers for parsing ini files.
 */
public enum IniTokens implements TokenType {

  SECTION(LexingMatcher.forPattern("^\\[([^\\]]+)\\]")),
  // note that KEY will also match SECTION, but it's lower priority so in practice it won't
  KEY(LexingMatcher.forPattern("^([^=\\s#;]+[^=\\s]*)(?=[\\t ]*=[\\t ]*)")),
  HASH_COMMENT(LexingMatcher.forLineComment("#"), true),
  SEMI_COMMENT(LexingMatcher.forLineComment(";"), true),
  WHITESPACE(LexingMatcher.forWhitespace("\n\r\t "), true),
  TAB(LexingMatcher.forChar('\t'), true),

  MULTILINE_VALUE(TokenMatchers.MULTILINE_VALUE),

  /**
   * Any text terminated by a line ending or EOF ( but don't consume EOF, as it
   * needs to be a distinct character)
   */
  VALUE(TokenMatchers.VALUE),
  EOF(LexingMatcher.forPattern("^$"));

  public static Tokens<IniTokens> tokens() {
    return new Tokens<>(
        IniTokens.class,
        IniTokens.EOF,
        List.of()
    );
  }

  private LexingMatcher matcher;

  @Getter
  private boolean whitespace;

  IniTokens(LexingMatcher matcher) {
    this(matcher, false);
  }

  IniTokens(LexingMatcher matcher, boolean whitespace) {
    this.matcher = matcher;
    this.whitespace = whitespace;
  }

  @Override
  public LexingMatcher matcher() {
    return matcher;
  }
}
