/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.ini;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import nz.org.riskscape.dsl.Lexer;
import nz.org.riskscape.dsl.Token;

/**
 * Parses an ini file in to an AST using RiskScape's DSL parsing code.  This implementation will improve over ini4j in
 * the following ways:
 *
 * * [x] Preserves absolute key-value ordering.  The wizard would have been easier to implement if we'd had this (and
 *       still might be in the future?) *
 * * [x] Preserve location information, e.g. bad value at line 12
 * * [x] Multi-line string support
 * * [ ] Whitespace preserving modifications - if we ever want to write out ini after modifying it, using the dsl
 *       framework is going to make it a lot easier to preserve the original author's whitespace formatting and
 *       comments.
 * * [ ] Keyword support?
 */
public class IniParser2 {

  private Lexer<IniTokens> lexer;

  public IniFile parse(String source) {
    lexer = new Lexer<>(IniTokens.tokens(), source);

    List<KeyValue> anonymous = Collections.emptyList();
    List<Section> sections = new LinkedList<>();

    while (!lexer.isEOF()) {
      if (lexer.peekType() == IniTokens.SECTION) {
        sections.add(parseSection());
      } else {
        anonymous = parseKeyValues();
      }

    }

    return new IniFile(anonymous, sections);
  }

  private List<KeyValue> parseKeyValues() {
    List<KeyValue> keyValues = new ArrayList<>();

    while (lexer.peekType() == IniTokens.KEY) {
      Token key = lexer.expect(IniTokens.KEY);
      Token value = lexer.expect(IniTokens.VALUE, IniTokens.MULTILINE_VALUE);

      keyValues.add(new KeyValue(key, value));
    }

    return keyValues;

  }

  private Section parseSection() {
    Token section = lexer.expect(IniTokens.SECTION);
    return new Section(section, parseKeyValues());
  }

}
