/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.typexp;

import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;

import lombok.RequiredArgsConstructor;
import lombok.ToString;
import nz.org.riskscape.dsl.Token;

/**
 * AST base type plus subtypes for type expressions
 */
@RequiredArgsConstructor @ToString
public abstract class AST {

  public final Token value;

  /**
   * A constant, literal value, like a piece of text or a number
   */
  public static class Constant extends AST {

    private final Object nativeValue;

    public Constant(Token value, Object nativeValue) {
      super(value);
      this.nativeValue = nativeValue;
    }

    /**
     * Return a Java representation of the parsed value
     * @return Object
     */
    public Object toNative() {
      return nativeValue;
    }
  }

  /**
   * A symbol is like an identifier - a string without quotes, a standalone
   * token.  Can represent a simple type, can also represent keys in a dictionary.
   */
  public static class Symbol extends AST {

    public Symbol(Token value) {
      super(value);
    }

    /**
     * The identifier for this symbol
     */
    public String ident() {
      return value.value;
    }
  }

  /**
   * A type that needs arguments given to it to be created, like a struct
   * has an identier and an argument list.
   */
  public static class ComplexType extends AST {
    public ComplexType(Token value, ValueList arguments) {
      super(value);
      this.arguments = arguments;
    }
    final ValueList arguments;

    public String ident() {
      return value.value;
    }

    public int argCount() {
      return arguments.values.size();
    }

    public AST arg(int i) {
      return arguments.values.get(i);
    }

    public List<AST> args() {
      return Lists.newArrayList(arguments.values);
    }
  }

  /**
   * Wraps a list of nodes so that it conforms to the AST interface
   */
  public static class ValueList extends AST {
    public final List<AST> values;

    public ValueList(Token value, List<AST> values) {
      super(value);
      this.values = values;
    }
  }

  /**
   * A map like structure where keys are symbols and the values are types or constants
   */
  public static class Dictionary extends AST {
    public final Map<Symbol, AST> values;

    public Dictionary(Token value, Map<Symbol, AST> values) {
      super(value);
      this.values = values;
    }

    public AST get(String identifier) {
      return values.entrySet()
          .stream()
          .filter(e -> e.getKey().ident().equals(identifier))
          .findFirst()
          .map(entry -> entry.getValue())
          .orElse(null);
    }

  }

}
