/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.typeset;

import java.util.Optional;

import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.ancestor.AncestorRule;
import nz.org.riskscape.engine.types.ancestor.AncestorType;
import nz.org.riskscape.engine.types.eqrule.Coercer;
import nz.org.riskscape.engine.types.eqrule.EquivalenceRule;
import nz.org.riskscape.engine.types.varule.Variance;
import nz.org.riskscape.engine.types.varule.VarianceRule;

/**
 * Minimal {@link TypeSet} interface for use when only testing of types is required/allowed.
 */
public interface TypeRules {

  /**
   * Search through the list of {@link EquivalenceRule}s and find the first one that can safely coerce a value from
   * the source type to the target type.  This will return an object that will allow the source type to be safely
   * converted to the target type.
   * @param sourceType the type of the value that is being assigned to something.
   * type
   * @param targetType the receiving type
   * @return a {@link Coercer}, or empty if none could be found.
   */
  Optional<Coercer> findEquivalenceCoercer(Type sourceType, Type targetType);

  /**
   * Search through the list of {@link VarianceRule}s to determine whether a value of sourceType can be assigned to
   * a receiver of targetType
   * @param sourceType the type of the value that is being assigned to something.
   * @param targetType the receiving type
   * @return a determined variance calculation, or {@link Variance#UNKNOWN} if it couldn't be determined.  You should
   * assume invariance if unknown is returned.
   */
  Variance testVariance(Type sourceType, Type targetType);

  /**
   * Compute the most common type between two types by comparing their ancestry using registered {@link AncestorRule}s
   *
   * The resulting type of this method is the type that can safely represent any value from either type.  If the value
   * needs some conversion first, then {@link AncestorType#isConverting()}  will return true.
   *
   * see nz.org.riskscape.engine.types.TypesTest for some concrete examples
   *
   * @return a {@link Type} that is the lowest common denominator of the two given types.  Gives empty if they have
   * nothing in common.
   */
  Optional<AncestorType> computeAncestorType(Type type1, Type type2);

  /**
   * @return true if values of `sourceType` can be assigned to values of `targetType`.  Note that this test is not
   * commutable, i.e. isAssignable(lhs, rhs) is not equivalent to isAssignable(rhs, lhs).
   */
  default boolean isAssignable(Type sourceType, Type targetType) {
    switch (testVariance(sourceType, targetType)) {
    case COVARIANT:
    case EQUAL:
      return true;
    default:
      return false;
    }
  }

  /**
   * Print some debug info to the TypeRules logger - this should be decorated with some extra context that helps put a
   * particular rule checks logging more easily interpreted
   */
  void debug(String msg, Object... args);
}
