/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.typeset;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import lombok.Getter;
import nz.org.riskscape.engine.Identified;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.WrappingType;

/**
 * Wraps a type to include an identifier and a {@link TypeSet} to link it back
 * to some managed and identified set of types among a set.
 * <p>
 * An {@link IdentifiedType} is equal iff the id, typeSet and subclass all match
 */
public abstract class IdentifiedType implements Identified, Type, WrappingType {

  /**
   * Strips off an outer IdentifiedType from the given type (if present)
   */
  public static Type strip(Type type) {
    if (type instanceof IdentifiedType) {
      return ((IdentifiedType) type).getUnderlyingType();
    } else {
      return type;
    }
  }

  @Getter
  private final String id;

  public IdentifiedType(String id) {
    this.id = id;
  }

  @Override
  public abstract Type getUnderlyingType();

  public abstract TypeSet getTypeSet();

  @Override
  public Object coerce(Object value) {
    return getUnderlyingType().coerce(value);
  }

  @Override
  public Class<?> internalType() {
    return getUnderlyingType().internalType();
  }

  @Override
  public <T extends Type> boolean isA(Class<T> type) {
    // special case : we allow sub class to be considered, it's an implementation detail of linked types
    return IdentifiedType.class.isAssignableFrom(type);
  }

  @Override
  public String toString() {
    return getId();
  }

  @Override
  public boolean equals(Object rhs) {
    if (rhs instanceof IdentifiedType) {
      IdentifiedType rhsType = (IdentifiedType) rhs;

      return rhsType.getTypeSet().equals(this.getTypeSet())
          && rhsType.getClass().equals(this.getClass())
          && rhsType.id.equals(this.id);
    } else {
      return false;
    }
  }

  @Override
  public int hashCode() {
    return this.id.hashCode();
  }

  @Override
  public int estimateSize(Object entry) {
    return getUnderlyingType().estimateSize(entry);
  }

  @Override
  public void toBytes(DataOutputStream os, Object toWrite) throws IOException {
    getUnderlyingType().toBytes(os, toWrite);
  }

  @Override
  public Object fromBytes(DataInputStream in) throws IOException {
    return getUnderlyingType().fromBytes(in);
  }
}
