/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.varule;

import java.util.Iterator;
import java.util.Objects;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.typeset.TypeRules;

/**
 * Will allow assignment if the source type is more specific than the target type.  Doesn't apply to source types
 * that aren't wrapping types.
 */
public class WrappingTypesRule implements VarianceRule {

  @Override
  public Variance test(TypeRules typeRules, Type source, Type target) {
    if (!(source.isWrapped() || target.isWrapped())) {
      return Variance.UNKNOWN;
    }

    Iterator<Type> sourceTypes = Lists.reverse(source.unwrap()).iterator();
    Iterator<Type> targetTypes = Lists.reverse(target.unwrap()).iterator();


    // The unwrapped type is in order of specificity, e.g like a sub type, so we check that each member of the rhs
    // exists at the same position on the lhs
    while (targetTypes.hasNext()) {
      Type targetPart = targetTypes.next();
      Type sourcePart = sourceTypes.hasNext() ? sourceTypes.next() : null;

      if (!Objects.equals(targetPart, sourcePart)) {

        if (targetPart == null) {
          return Variance.COVARIANT;

        } else if (sourcePart == null) {
          // target is more specific
          return Variance.INVARIANT;
        } else if (sourcePart.equals(source) && targetPart.equals(target)) {
          // Type#unwrap includes the wrapping type in the returned list. We we don't break out here
          // we get infinite recursion because we'd be trying to test the variance with the same inputs.
          return Variance.UNKNOWN;
        } else {
          // it's possible that the target and source might be complex types, in which case they could be variant, but
          // not equals
          return typeRules.testVariance(sourcePart, targetPart);
        }
      }
    }

    // all parts are equals - this probably should be picked up by the equality part...
    return Variance.COVARIANT;
  }

}
