/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.eqrule;

import java.util.Optional;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Geom;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.typeset.TypeRules;

/**
 * {@link EquivalenceRule} that allows a {@link Struct} with a single {@link Geom} member to be coerced to
 * a {@link Geom}.  This allows expressions like `intersects(asset, region)` instead of
 * `intersects(asset.geom_member, region.geom)`.  This is really helpful with feature type structs to allow geometry
 * operations to be applied to them.
 *
 * NB we might want this to apply more generally, e.g. any struct with a single member of type X can be coerced to X, or
 * consider a special geometry wrapping type to signal that it's a special feature geometry attribute.
 */
public class StructGeometryRule implements EquivalenceRule {

  @Override
  public Optional<Coercer> getCoercer(TypeRules typeRules, Type sourceType, Type targetType) {

    if (!typeRules.isAssignable(targetType, Types.GEOMETRY)) {
      return Optional.empty();
    }

    Struct sourceStruct = sourceType.find(Struct.class).orElse(null);

    if (sourceStruct == null) {
      return Optional.empty();
    }

    StructMember geomMember = null;
    Optional<Coercer> geomCoercer = Optional.empty();
    for (StructMember member : sourceStruct.getMembers()) {

      Optional<Coercer> coercer;
      if (typeRules.isAssignable(member.getType(), Types.GEOMETRY)) {
        coercer = Optional.of(Coercer.identity(member.getType()));
      } else if (member.getType().find(Struct.class).isPresent()) {
        // Don't descend into the Structure
        coercer = Optional.empty();
      } else {
        coercer = typeRules.findEquivalenceCoercer(member.getType(), Types.GEOMETRY);
      }

      if (!coercer.isPresent()) {
        continue;
      }

      if (geomMember == null) {
        geomMember = member;
        geomCoercer = coercer;
      } else {
        // two geometry members - ambiguous, don't allow it
        return Optional.empty();
      }
    }

    if (geomMember == null) {
      return Optional.empty();
    } else {
      StructMember foundMember = geomMember;
      Coercer foundCoercer = geomCoercer.get();

      return Optional.of(Coercer.build(sourceType, foundCoercer.getTargetType(), sourceValue -> {
        Tuple tuple = (Tuple) sourceValue;
        return foundCoercer.apply(tuple.fetch(foundMember));
      }));
    }
  }

}
