/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.eqrule;

import java.util.Optional;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.typeset.TypeRules;

/**
 * Allows assignment between structs and simple values if the struct has a single member who's type matches the single
 * value.  Allows recursive equivalence between the member of the struct and the simple value.
 *
 * NB need to consider how to open up the non-struct type to be not just a simple type...
 */
public class SingleValueStructRule implements EquivalenceRule {

  @Override
  public Optional<Coercer> getCoercer(TypeRules typeRules, Type sourceType, Type targetType) {
    if (singleMemberStruct(sourceType)) {
      Struct sourceStruct = sourceType.find(Struct.class).get();
      Struct targetStruct = targetType.find(Struct.class).orElse(null);

      // special case: a wrapping type might be hiding the fact that we're wrapping the same types - if we match, we
      // end up building a coercer that goes from {a: b} => b => {c: b}
      if (sourceStruct == targetStruct) {
        return Optional.empty();
      }

      Type memberType = sourceStruct.getMembers().get(0).getType();

      if (typeRules.isAssignable(memberType, targetType)) {
        return structToValue(sourceType, memberType, null);
      } else {
        return typeRules.findEquivalenceCoercer(memberType, targetType)
            .map(subCoercer -> structToValue(sourceType, subCoercer.getTargetType(), subCoercer).get());
      }
    } else if (singleMemberStruct(targetType)) {
      Struct targetStruct = targetType.find(Struct.class).get();
      Struct sourceStruct = sourceType.find(Struct.class).orElse(null);

      // special case: a wrapping type might be hiding the fact that we're wrapping the same types - if we match, we
      // end up building a coercer that goes from {a: b} => b => {c: b}
      if (sourceStruct == targetStruct) {
        return Optional.empty();
      }
      Type memberType = targetStruct.getMembers().get(0).getType();

      if (typeRules.isAssignable(sourceType, memberType)) {
        return valueToStruct(sourceType, targetType, targetStruct, null);
      } else {
        return typeRules.findEquivalenceCoercer(sourceType, memberType)
            .map(subCoercer -> valueToStruct(sourceType, targetType, targetStruct, subCoercer).get());
      }
    } else {
      return Optional.empty();
    }
  }

  private Optional<Coercer> valueToStruct(Type source, Type target, Struct targetStruct, Coercer subCoercer) {
    return Optional.of(Coercer.build(source, target, value ->
      Tuple.ofValues(targetStruct, subCoercer == null ? value : subCoercer.apply(value))));
  }

  private Optional<Coercer> structToValue(Type source, Type target, Coercer subCoercer) {
    return Optional.of(Coercer.build(source, target, tuple -> {
      Object rawValue = ((Tuple) tuple).fetch(0);
      if (subCoercer == null) {
        return rawValue;
      } else {
        return subCoercer.apply(rawValue);
      }
    }));
  }

  private boolean singleMemberStruct(Type type) {
    return type.find(Struct.class).map(Struct::size).orElse(0) == 1;
  }

}
