/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.types.eqrule;

import java.util.Optional;

import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;

import nz.org.riskscape.engine.types.GeomType;
import nz.org.riskscape.engine.types.MultiGeom;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.typeset.TypeRules;

/**
 * Allows a singular specific geometry type (LINE, POINT, POLYGON) to be coerced to a multi form of
 * the same type.
 */
@SuppressWarnings("unchecked")
public class SingleGeomToMultiGeomRule implements EquivalenceRule {

  @Override
  public Optional<Coercer> getCoercer(TypeRules typeRules, Type sourceType, Type targetType) {
    Optional<MultiGeom> multiTarget = targetType.find(MultiGeom.class);
    if (! multiTarget.isPresent()) {
      // target is not a multi geom, we're out.
      return Optional.empty();
    }
    Type containedTargetType = multiTarget.get().getContainedType();

    if (sourceType.find(GeomType.Line.class).isPresent()
        && typeRules.isAssignable(sourceType, containedTargetType)) {
      return Optional.of(Coercer.build(sourceType, targetType, x -> {
        LineString line = (LineString)x;
        return line.getFactory().createMultiLineString(new LineString[]{line});
      }));
    }
    if (sourceType.find(GeomType.Point.class).isPresent()
        && typeRules.isAssignable(sourceType, containedTargetType)) {
      return Optional.of(Coercer.build(sourceType, targetType, x -> {
        Point point = (Point)x;
        return point.getFactory().createMultiPoint(new Point[]{point});
      }));
    }
    if (sourceType.find(GeomType.Polygon.class).isPresent()
        && typeRules.isAssignable(sourceType, containedTargetType)) {
      return Optional.of(Coercer.build(sourceType, targetType, x -> {
        Polygon polygon = (Polygon)x;
        return polygon.getFactory().createMultiPolygon(new Polygon[]{polygon});
      }));
    }
    return Optional.empty();
  }

}
