/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.eqrule;

import java.util.Optional;

import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.typeset.TypeRules;

/**
 * Allows equivalence to be determined by the types inside a nullable
 */
public class NullableRule implements EquivalenceRule {

  @Override
  public Optional<Coercer> getCoercer(TypeRules typeRules, Type sourceType, Type targetType) {
    if (targetType.isNullable()) {
      Type denulledTarget = Nullable.strip(targetType);
      Type denulledSource = Nullable.strip(sourceType);

      return typeRules
          .findEquivalenceCoercer(denulledSource, denulledTarget)
          // Don't use the given targetType in case the denulled coercer advertises a different target type, instead
          // we rebuild the type as it was found (not as given in the args). This might be where we have a wrapping
          // type around a member type which we don't want to lose after coercion, e.g give me a coercer from
          // {x: referenced(point, 'epsg:2193')} to geom - we don't want to lose the fact that the geom we've found
          // is a point in epsg:2193.
          .map(child -> Coercer.build(sourceType, Nullable.of(child.getTargetType()), input -> {
            if (input == null) {
              // we can't pass null to the child coercer that will not be expecting it. so we short
              // circuit and just return null ourselves.
              return null;
            }
            return child.apply(input);
          }));
    } else {
      return Optional.empty();
    }
  }

}
