/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.ancestor;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructBuilder;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.typeset.TypeRules;

public class StructMemberAncestorRule extends TypeSafeAncestorRule<Struct, Struct> {

  @Override
  protected Optional<AncestorType> computeSafeAncestor(TypeRules typeRules, Struct type1, Struct type2) {

    /*
     * Implementation note:
     *
     * We could make this closer to the union step and give nullable members when members are missing or have no common
     * ancestor, but we don't need it right now.
     */
    if (new HashSet<>(type1.getMemberKeys()).equals(new HashSet<>(type2.getMemberKeys()))) {
      List<AncestorType> memberAncestors = new ArrayList<>(type1.size());

      StructBuilder structBuilder = Struct.builder();
      for (StructMember member1 : type1.getMembers()) {
        // should always be here - we've checked the key sets are the same
        StructMember member2 = type2.getMember(member1.getKey()).get();
        AncestorType memberAncestor =
            typeRules.computeAncestorType(member1.getType(), member2.getType()).orElse(null);

        if (memberAncestor == null) {
          return Optional.empty();
        }

        memberAncestors.add(memberAncestor);
        structBuilder.add(member1.getKey(), memberAncestor.getType());
      }

      // should always be here - we've check the key sets are the samectBuilder.build();
      Struct ancestorStruct = structBuilder.build();
      AncestorType.Converter converter = (source) -> {
        Tuple input = (Tuple) source;
        Tuple output = Tuple.of(ancestorStruct);
        Iterator<AncestorType> iterator = memberAncestors.iterator();
        int index = 0;
        for (Object value : input.getValues()) {
          output.set(index++, iterator.next().getConvert().apply(value));
        }

        return output;
      };
      return Optional.of(AncestorType.of(ancestorStruct, converter));

    } else {
      return Optional.empty();
    }
  }

}
