/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.ancestor;

import java.util.Optional;

import org.geotools.referencing.CRS;
import org.locationtech.jts.geom.Envelope;

import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.typeset.TypeRules;

/**
 * Allows two Referenced geometry types to be ancestors, maintaining their {@link Referenced} wrapping type so long as
 * they are in the same CRS.
 */
public class ReferencedAncestorRule extends TypeSafeAncestorRule<Referenced, Referenced> {

  @Override
  public Optional<AncestorType> computeSafeAncestor(TypeRules typeRules, Referenced lhs, Referenced rhs) {
    return typeRules.computeAncestorType(lhs.getUnderlyingType(), rhs.getUnderlyingType()).map(ancestor -> {
      // TODO we could support a converting version that does a reprojection...
      if (!CRS.equalsIgnoreMetadata(lhs.getCrs(), rhs.getCrs())) {
        return ancestor;
      } else {
        return ancestor.withNewType(
            // expand the
          new Referenced(ancestor.getType(), lhs.getCrs(), combineEnvelopes(lhs.getEnvelope(), rhs.getEnvelope()))
        );
      }
    });
  }
  private Envelope combineEnvelopes(Envelope a, Envelope b) {
    if (a == null || b == null) {
      // If either bounds are unknown, then we won't falsely advertise the bounds we do know, as it'll be incomplete
      return null;
    }
    Envelope combinedEnvelope = new Envelope();
    combinedEnvelope.expandToInclude(a);
    combinedEnvelope.expandToInclude(b);
    return combinedEnvelope;
  }

}
