/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.ancestor;

import java.util.function.BiFunction;
import java.util.function.Function;

import lombok.Data;
import nz.org.riskscape.engine.types.Type;

/**
 * The result of computing an ancestor for two types.  Might be just a new type, or might be a type plus a conversion.
 * If there is a conversion ({@link #isConverting()} is true) then the {@link #getConvert()} function must be applied
 * to any values before they are said to belong to the resulting type.
 */
@Data
public class AncestorType {

  /**
   * Wrapper around {@link Function} to avoid nasty generics all over the shop
   */
  public interface Converter extends Function<Object, Object> {
  }

  /**
   * No conversion
   */
  public static final Converter IDENTITY = v -> v;

  /**
   * The resulting ancestor type
   */
  private final Type type;

  /**
   * The function used to convert descendants to the given type
   */
  private final Converter convert;

  public static AncestorType of(Type type, Converter convert) {
    return new AncestorType(type, convert);
  }

  /**
   * Create an {@link AncestorType} with no conversion
   */
  public static AncestorType of(Type type) {
    return new AncestorType(type, IDENTITY);
  }

  /**
   * Return a new {@link AncestorType} with the same conversion function, but a new type
   */
  public AncestorType withNewType(Type newType) {
    return new AncestorType(newType, convert);
  }

  /**
   * Return a new {@link AncestorType} with the same type, but with a converter that composes the functionality of
   * the existing convert functionality
   * @param compose a function that should convert values for this {@link AncestorType}, where the original convert
   * function is passed in for you to use (or ignore) as you see fit.
   */
  public AncestorType compose(BiFunction<Converter, Object, Object> compose) {
    return new AncestorType(type, (value) -> {
      return compose.apply(this.convert, value);
    });
  }

  /**
   * @return true if this {@link AncestorType} requires conversion for the ancestor type to be used.
   */
  public boolean isConverting() {
    return !convert.equals(IDENTITY);
  }

}
