/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Wrapping types add extra restrictions or metadata to a type that narrows that type.  A wrapping type should not widen
 * the type, e.g. make more values possible.  A good example is something like {@link WithinSet} or an
 * {@link Enumeration}, which wrap types to make only particular values of it valid.
 */
public interface WrappingType extends Type {

  /**
   * @return the type that we immediately wrap.  Used by {@link Type#getUnwrappedType()} to descend through all wrapped
   * {@link Type} to find the first unwrapped one.
   */
  Type getUnderlyingType();

  @Override
  default Struct asStruct() {
    // FIXME: This is plain wrong, but 'fixing it' causes a bunch of test failures that all need investigating
    // return Struct.of(SimpleType.DEFAULT_MEMBER_NAME, this);
    return getUnderlyingType().asStruct();
  }

  @Override
  default int estimateSize(Object entry) {
    return getUnderlyingType().estimateSize(entry);
  }

  @Override
  default void toBytes(DataOutputStream os, Object toWrite) throws IOException {
    getUnderlyingType().toBytes(os, toWrite);
  }

  @Override
  default Object fromBytes(DataInputStream in) throws IOException {
    return getUnderlyingType().fromBytes(in);
  }

  @Override
  default boolean isNullable() {
    return getUnderlyingType().isNullable();
  }

  @Override
  default <T, U> U visit(TypeVisitor<T, U> tv, T data) {
    return tv.compoundType(this, TypeVisitor.children(getUnwrappedType()), data);
  }
}
