/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.types;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import lombok.EqualsAndHashCode;
import lombok.Getter;

/**
 * Wrapping type that stores extra metadata along with a given type. For example,
 * this lets you potentially store the expression that created the type (e.g.
 * `bookmark('foo')`), along with the type itself (e.g. {@link RelationType}.
 *
 * Multiple metadata objects can be associated with the underlying type. We assume
 * that each metadata object is of a unique class, but this is not enforced at all.
 */
@EqualsAndHashCode
public class WithMetadata implements WrappingType {

  /**
   * Creates a new WithMetadata that wraps the given type and associates the metadata with it
   */
  public static Type wrap(Type given, Object... metadata) {
    // note that if the type being wrapped is null, we shuffle things round so the Nullable is on the outside
    return Nullable.rewrap(given, underlyingType -> new WithMetadata(underlyingType, Arrays.asList(metadata)));
  }

  /**
   * @return metadata matching the class given, if any
   */
  public static <T> Optional<T> find(Type type, Class<T> metadataClass) {
    return type.findAllowNull(WithMetadata.class)
        .flatMap(t -> t.getMetadata(metadataClass));
  }

  @Getter
  private final Type underlyingType;

  private final List<?> metadata;

  private WithMetadata(Type wrapped, List<?> metadata) {
    if (wrapped.find(WithMetadata.class).isPresent()) {
      // TODO this requires more thought if we ever hit it. We could just combine the metadata,
      // but that might result in odd behaviour, e.g. if there are 2 different Bookmark objects
      // then which one should be returned?
      throw new UnsupportedOperationException("Cannot have multiple nested WithMetadata types");
    }
    this.underlyingType = wrapped;
    this.metadata = metadata;
  }

  /**
   * @return metadata of the given target class, if any. If there are multiple metadata
   *         objects that match (e.g. Identified), then the first match found is returned
   */
  public <T> Optional<T> getMetadata(Class<T> target) {
    for (Object entry : metadata) {
      if (target.isInstance(entry)) {
        return Optional.of(target.cast(entry));
      }
    }
    return Optional.empty();
  }

  @Override
  public Object coerce(Object value) throws CoercionException {
    return getUnderlyingType().coerce(value);
  }

  @Override
  public Class<?> internalType() {
    return getUnderlyingType().internalType();
  }

  @Override
  public String toString() {
    return "WithMetadata[" + underlyingType.toString() + "]";
  }
}
