/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import com.google.common.collect.ImmutableMap;

import lombok.EqualsAndHashCode;
import lombok.Getter;

/**
 * Represents any java object that has been coerced to the associated type.
 * @param type to coerce to.
 */
@EqualsAndHashCode
public class Value {

  public static Value of(Type type, Object rawValue) throws CoercionException {
    return new Value(type, rawValue);
  }

  /**
   * Creates a new value.
   * @param type the type that the value must conform to
   * @param rawValue unchecked value, that will be coerced before this object is fully created
   */
  public Value(Type type, Object rawValue) throws CoercionException {
    this.type = type;
    this.value = type.coerce(rawValue);
  }

  /**
   * The underlying value, matches {@link #getType()}.
   */
  @Getter
  private final Object value;

  /**
   * The type that represents {@link #getValue()}
   */
  @Getter
  private final Type type;

  @Override
  public String toString() {
    return String.format("%s:%s", type, value);
  }

  /**
   * @return If the unwrapped type of this value is not a struct, then this
   * method will return a structed version of the value where, value is the only
   * member of the struct
   */
  public Value toStruct() {
    Type unwrapped = type.getUnwrappedType();
    if (unwrapped instanceof Struct) {
      return new Value(unwrapped, value);
    } else if (unwrapped instanceof SimpleType) {
      SimpleType simple = (SimpleType) unwrapped;
      return new Value(simple.asStruct(), ImmutableMap.of("value", value));
    } else {
      throw new RuntimeException(String.format("Can not represent %s as a struct - unknown type %s", value, unwrapped));
    }
  }

}

