/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Optional;

import org.locationtech.jts.geom.Geometry;

import nz.org.riskscape.problem.ProblemException;

/**
 * Singleton instances of SimpleValued types to avoid needless construction
 * TODO maybe make the constructors package level to hammer it home?
 */
public class Types {

  public static final Text TEXT = SimpleType.getInstance(Text.class);
  public static final Integer INTEGER = SimpleType.getInstance(Integer.class);
  public static final Floating FLOATING = SimpleType.getInstance(Floating.class);
  public static final Smallfloat SMALLFLOAT = SimpleType.getInstance(Smallfloat.class);
  public static final Decimal DECIMAL = SimpleType.getInstance(Decimal.class);
  public static final Anything ANYTHING = SimpleType.getInstance(Anything.class);
  public static final Geom GEOMETRY = SimpleType.getInstance(Geom.class);
  public static final GeomType.Line LINE = SimpleType.getInstance(GeomType.Line.class);
  public static final GeomType.Point POINT = SimpleType.getInstance(GeomType.Point.class);
  public static final GeomType.Polygon POLYGON = SimpleType.getInstance(GeomType.Polygon.class);
  public static final Bool BOOLEAN = SimpleType.getInstance(Bool.class);
  public static final Date DATE = SimpleType.getInstance(Date.class);
  public static final Nothing NOTHING = SimpleType.getInstance(Nothing.class);

  public static Type fromJavaType(Class<?> javaType) {
    Type fromJavaTypeInternal = fromJavaTypeInternal(javaType);

    if (fromJavaTypeInternal == null) {
      throw new RuntimeException("Unmapped type: " + javaType);
    } else {
      return fromJavaTypeInternal;
    }
  }

  public static Optional<Type> fromJavaTypeOptional(Class<?> javaType) {
    return Optional.ofNullable(fromJavaTypeInternal(javaType));
  }

  private static Type fromJavaTypeInternal(Class<?> javaType) {
    if (javaType.equals(java.lang.Integer.class) || javaType.equals(int.class)
        || javaType.equals(java.lang.Long.class) || javaType.equals(long.class)
        || javaType.equals(java.lang.Short.class) || javaType.equals(short.class)
        || javaType.equals(java.lang.Byte.class)|| javaType.equals(byte.class)) {
      return Types.INTEGER;
    } else if (javaType.equals(java.lang.Boolean.class) || javaType.equals(boolean.class)
        || javaType.equals(java.lang.Boolean.TYPE)) {
      return Types.BOOLEAN;
    } else if (javaType.equals(String.class)) {
      return Types.TEXT;
    } else if (Geometry.class.isAssignableFrom(javaType)) {
      return Types.GEOMETRY;
    } else if (javaType.equals(BigDecimal.class) || javaType.equals(BigInteger.class)) {
      return Types.DECIMAL;
    } else if (javaType.equals(Double.class) || javaType.equals(double.class)) {
      return Types.FLOATING;
    } else if (javaType.equals(Float.class) || javaType.equals(float.class)) {
      return Types.FLOATING;
    } else if (java.util.Date.class.isAssignableFrom(javaType)) {
      return Types.DATE;
    }

    return null;
  }

  /**
   * Variant of fromJavaTypeOptional that strictly maps to the internal java type.  Only works for the core Riskscape
   * types.
   */
  public static Optional<Type> fromStrictJavaTypeOptional(Class<?> javaType) {
    Type found;
    if (javaType.equals(java.lang.Long.class)) {
      found = Types.INTEGER;
    } else if (javaType.equals(java.lang.Boolean.class)) {
      found = Types.BOOLEAN;
    } else if (javaType.equals(String.class)) {
      found = Types.TEXT;
    } else if (Geometry.class.isAssignableFrom(javaType)) {
      found = Types.GEOMETRY;
    } else if (javaType.equals(BigDecimal.class)) {
      found = Types.DECIMAL;
    } else if (javaType.equals(Double.class)) {
      found = Types.FLOATING;
    } else if (java.util.Date.class.isAssignableFrom(javaType)) {
      found = Types.DATE;
    } else {
      found = null;
    }

    return Optional.ofNullable(found);
  }

   /**
   * Convenience function to codify standard behaviour for when type.find() fails to find something and you want to
   * fail fast in a {@link ProblemException#catching(nz.org.riskscape.problem.ProblemException.Call)} block
   * @throws ProblemException with a {@link nz.org.riskscape.engine.types.TypeProblems#get()}
   */
  @SuppressWarnings("unchecked")
  public static <T extends Type> T findOrThrow(Object context, T wanted, Type actual) throws ProblemException {
    // the api wants a class, but the error needs an instance of a type
    return (T) actual.find(wanted.getClass()).orElseThrow(() -> new ProblemException(
        TypeProblems.get().mismatch(context, wanted, actual)));
  }

}
