/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.util.Arrays;
import java.util.List;
import java.util.function.BiConsumer;

import nz.org.riskscape.engine.typexp.TypeConstructor;
import nz.org.riskscape.engine.util.Pair;

/**
 * A Visitor interface for descending through an arbitrary type 'tree' - the other half of
 * {@link Type#visit(TypeVisitor, Object)}.  Allows calling code to traverse through atomic and compound types without
 * having to know in advance about every single type class.
 *
 * For the purposes of this interface, an atomic types is defined as one whose identity is not defined by other types
 * and where it is not possible to recurse in to it to visit other types.  A "compound type" is one whose identity is
 * based on other sub-types, and so can be visited with the visitor.
 *
 * Note that these concepts are quite specific to the visitor and are mostly orthogonal to concepts like a
 * {@link SimpleType} or the two construction methods available on a {@link TypeConstructor}
 *
 * Types are
 * required to implement {@link Type#visit(TypeVisitor, Object)}, which should in turn call either
 * {@link #atomicType(Type, Object)} or {@link #compoundType(Type, List, Object)}, depending on which sort of
 * {@link Type} it is, e.g. does the type declare any other types that are part of its identity.
 *
 * Compound types should pass a list of all the types that form part of their identity to the `compoundType` visiting
 * method, along with some optional metadata for each child.  The metadata can be used to disambiguate children
 * and provide context where required, the {@link #NO_META} object is available of a child does not need any metadata.
 * There is no need to provide context that is not already implicit based on the parent type.
 *
 * The {@link TypeVisitor} interface does not control the flow of visiting - an implementor might decide to implement
 * a DFS or a BFS, and is not obliged or forced (via the API) to descend in to all types that are part of the tree.
 *
 *
 * @param <T> A data type to pass around through calls.
 * @param <U> A type that visiting methods must return.
 */
public interface TypeVisitor<T, U> {

  /**
   * Null-alternative when the child of a compound type has no metadata
   */
  Object NO_META = new Object();

  /**
   * Convenience function for doing a breadth first search of a type, passing the given visit data to each seen type.
   * Lets you do simple one liners, e.g. `List<Type> types = bfs(someType, new LinkedList<Type>(), (l, t) -> l.add(t))`
   */
  static <T> T bfs(Type root, T visitData, BiConsumer<T, Type> consumer) {
    root.visit(new TypeVisitor<T, Void>() {

      @Override
      public Void atomicType(Type atomicType, T data) {
        consumer.accept(data, atomicType);
        return null;
      }

      @Override
      public Void compoundType(Type compoundType, List<Pair<Type, ?>> children, T data) {
        consumer.accept(data, compoundType);
        for (Pair<Type, ?> pair : children) {
          pair.getLeft().visit(this, data);
        }
        return null;
      }
    }, visitData);

    return visitData;
  }

  /**
   * Handy list constructor for a single child type with no metadata
   */
  static List<Pair<Type, ?>> children(Type type0) {
    return children(type0, NO_META);
  }

  /**
   * Handy list constructor for a single child type with metadata
   */
  static List<Pair<Type, ?>> children(Type type0, Object meta0) {
    return Arrays.asList(Pair.of(type0, meta0));
  }

  /**
   * Handy list constructor for two child types with no metadata
   */
  static List<Pair<Type, ?>> children(Type type0, Type type1) {
    return children(type0, NO_META, type1, NO_META);
  }

  /**
   * Handy list constructor for two child types with metadata
   */
  static List<Pair<Type, ?>> children(Type type0, Object meta0, Type type1, Object meta1) {
    return Arrays.asList(Pair.of(type0, meta0), Pair.of(type1, meta1));
  }

  /**
   * Visit an atomic type - there is no further recursion possible here
   */
  U atomicType(Type atomicType, T data);

  /**
   * Visit a compound type, allowing possible recursion via the list of children.  It is up to the implementor to do
   * any recursive visiting by calling the `visit` method on any children and also to decide/declare traversal order.
   */
  U compoundType(Type compoundType, List<Pair<Type, ?>> children, T data);

}
