/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.util.List;
import java.util.Set;

import nz.org.riskscape.dsl.TokenType;
import nz.org.riskscape.engine.i18n.Message;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.rl.ast.Expression;

public interface TypeProblems extends ProblemFactory {

  // Struct was declared to have <expected-member>, but this was not provided in <actual-struct>
  Problem structMemberNotProvided(StructMember expected, Struct actual);

  /**
   * Struct must have at least one member (non-empty)
   */
  Problem structMustBeNonEmpty(Object context);

  /**
   * Struct member named `member` must be of type `expected`
   */
  Problem structMemberMustBeOfType(Object context, String member, Type expected, Type actual);

  /**
   * Struct must have a single <requiredType> member, but none were found among <actual>
   */
  Problem structMustHaveMemberType(Type requiredType, Struct actual);

  // Expected <thing> to be a struct, but got <actual>
  Problem notStruct(Expression thing, Type actual);

  // Type mismatch for <thing>. Expected '<expected>' but found '<actual>
  Problem mismatch(Object affected, Type expected, Type actual);

  Problem mismatch(Object affected, Class<?> expected, Class<?> actual);

  Problem mismatch(Object affected, TokenType expected, TokenType actual);

  @Message("Expected the items in the list to be a consistent ''{0}'' type, but got ''{1}'' type instead. "
      + "Note that mixing Integer and Floating values in the same list will result in ''Anything'' type")
  Problem listItemMismatch(Class<?> expected, Type actual);

  /**
   * When affected requires one of allowedTypes, but actual was provided.
   * @param affected
   * @param allowedTypes  list of types that would be accepted
   * @param actual        the type that was provided
   * @return Problem
   */
  Problem requiresOneOf(Object affected, List<Type> allowedTypes, Type actual);

  Problem couldNotCoerce(Type given, Type expected);

  Problem failedToValidate(String typeId, Resource resource);

  Problem duplicateKeys(Set<StructMember> duplicates);

  /**
   * Variant of {@link #mismatch(Object, Type, Type) specific for nullability issues.
   */
  @Message("{0} cannot be a nullable type. Expected {1} type, but got {2}")
  Problem cannotBeNull(Object thing, Type expected, Type actual);

  static TypeProblems get() {
    return Problems.get(TypeProblems.class);
  }
}
