/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.util.Optional;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Identified;
import nz.org.riskscape.engine.io.NoSuitableTypeIOException;
import nz.org.riskscape.engine.io.TypeIO;
import nz.org.riskscape.engine.io.TypeIOConstructor;
import nz.org.riskscape.engine.typexp.TypeConstructor;

/**
 * Meta-class for riskscape types that allows types to be plugged in to the riskscape engine.  Extra information that
 * we would be adding to the class object if Java allowed that.
 */
@RequiredArgsConstructor()
public class TypeInformation implements Identified {

  /**
   * An identifier for the type - should be ([a-z]) so that it can be used to identify types with riskscape's type
   * expression language
   */
  @Getter
  private final String id;

   /**
   * The exact type that results from {@link TypeConstructor} will be of.  Conceptually, this is the class that we are
   * extending/augmenting with extra meta-information.
   */
  @Getter
  private final Class<? extends Type> typeClass;

  /**
   * The java type that is used to hold values that conform to this type.  Note that multiple riskscape types might use
   * the same java type to represent their values.
   */
  @Getter
  private final Class<?> valueType;

  /**
   * A {@link TypeConstructor} that can construct instances of this type using riskscape's type expression language.
   * Note that this produces instances of {@link Type}, not values that conform to this type.
   */
  @Getter
  private final TypeConstructor constructor;


  /**
   * An optional {@link TypeIOConstructor} that can return/build {@link TypeIO} objects for a particular type.  For
   * simple types, this will typically be the same instance.  For complex types, it will be a {@link TypeIO} object
   * that works for a particular type.
   *
   * Not providing a {@link TypeIOConstructor} for a {@link Type} will potentially result in
   * {@link NoSuitableTypeIOException}s being thrown from engine code - all going well these will be handled and given
   * back to the user with help on how to avoid the error.
   */
  @Getter
  private final Optional<TypeIOConstructor> typeIOConstructor;

  public TypeInformation(
      String id,
      Class<? extends Type> typeClass,
      Class<?> valueType,
      TypeConstructor constructor
  ) {
    this(id, typeClass, valueType, constructor, Optional.empty());
  }


}
