/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import lombok.Getter;
import lombok.RequiredArgsConstructor;

/**
 * A {@link LambdaType} that also includes the scope.  Mostly used to facilitate the mechanics of realizing the
 * body of a lambda expression during function realization.
 */
@RequiredArgsConstructor
public class ScopedLambdaType implements Type {

  private final Struct scopedAsStruct = Struct.of(SimpleType.DEFAULT_MEMBER_NAME, this);

  // for now, I haven't added this type to the type registry as I can't see a case for it - it's mostly an internal
  // detail to support function realization
  public ScopedLambdaType(Struct struct, String... args) {
   this(struct, Arrays.asList(args));
  }

  /**
   * The scope (or input type) that was closed over when the expression that contained this lambda expression was
   * realized
   */
  @Getter
  private final Struct scopeType;

  /**
   * The names of the arguments that are passed to the expression.  Once a Lambda is realized, these names become
   * part of the type's identity and so matter for equality
   */
  @Getter
  private final List<String> args;

  /**
   * @return the number of arguments the expression declared.
   */
  public int getArity() {
    return this.args.size();
  }

  @Override
  public Struct asStruct() {
    return scopedAsStruct;
  }

  @Override
  public Object coerce(Object value) throws CoercionException {
    throw new UnsupportedOperationException();
  }

  @Override
  public Class<?> internalType() {
    return Object.class;
  }

  @Override
  public int estimateSize(Object entry) {
    return -1;
  }

  @Override
  public void toBytes(DataOutputStream os, Object toWrite) throws IOException {
    throw new UnsupportedOperationException();
  }

  @Override
  public Object fromBytes(DataInputStream in) throws IOException {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean equals(Object obj) {
    if (obj instanceof ScopedLambdaType) {
      ScopedLambdaType rhs = (ScopedLambdaType) obj;
      return
          rhs.scopeType.equals(this.scopeType)
          // once the type is scoped, the arg names now matter for equality
          && rhs.getArgs().equals(this.getArgs());
    } else {
      return false;
    }
  }

  @Override
  public int hashCode() {
    return Objects.hash(this.scopeType, this.args);
  }

  @Override
  public String toString() {
    return String.format("λ-Scoped(%s, scope=%s)",
        this.args.stream().collect(Collectors.joining(",")), scopeType);
  }

  @Override
  public <T, U> U visit(TypeVisitor<T, U> tv, T data) {
    return tv.compoundType(this, TypeVisitor.children(scopeType), data);
  }

  /**
   * Builds a scope that includes the lambda args (based on the actual types that
   * the lambda expression is realized with), and returns the scope type.
   *
   * @param lambdaArgTypes the actual types of the lambda arguments
   * @return a Struct combining {@link #getScopeType()} and lambdaArgTypes
   */
  public Struct buildCallingScopeType(Type... lambdaArgTypes) {
    Struct newScopeType = scopeType;

    // replace the lambda arg with the actual values to use
    for (int i = 0; i < lambdaArgTypes.length; i++) {
      String lambdaArgName = args.get(i);
      // NB at the moment, we replace the previous member if it existed.
      // In the future we might want to allow 'parent scope', but replacing is good
      // enough for now and probably will get us through "just fine"
      newScopeType = newScopeType.addOrReplace(lambdaArgName, lambdaArgTypes[i]);
    }
    return newScopeType;
  }

}
