/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;

import nz.org.riskscape.engine.typexp.AST.Symbol;
import nz.org.riskscape.engine.typexp.SimpleTypeConstructor;
import nz.org.riskscape.engine.typexp.TypeBuilder;

import lombok.Getter;
import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor
public class MultiGeom implements WrappingType, ContainingType {

  public static final MultiGeom MULTI_LINE = new MultiGeom(MultiLineString.class,
      SimpleType.getInstance(GeomType.Line.class));
  public static final MultiGeom MULTI_POINT = new MultiGeom(MultiPoint.class,
      SimpleType.getInstance(GeomType.Point.class));
  public static final MultiGeom MULTI_POLYGON = new MultiGeom(MultiPolygon.class,
      SimpleType.getInstance(GeomType.Polygon.class));

  public static final SimpleTypeConstructor MULTI_LINE_TYPE_CONSTRUCTOR =
      (TypeBuilder typeBuilder, Symbol symbol) -> {
    return MULTI_LINE;
  };

  public static final TypeInformation MULTI_LINE_TYPE_INFORMATION = new TypeInformation(
      "multiline",
      MultiGeom.class,
      MultiLineString.class,
      MULTI_LINE_TYPE_CONSTRUCTOR
  );

  public static final SimpleTypeConstructor MULTI_POINT_TYPE_CONSTRUCTOR =
      (TypeBuilder typeBuilder, Symbol symbol) -> {
    return MULTI_POINT;
  };

  public static final TypeInformation MULTI_POINT_TYPE_INFORMATION = new TypeInformation(
      "multipoint",
      MultiGeom.class,
      MultiPoint.class,
      MULTI_POINT_TYPE_CONSTRUCTOR
  );

  public static final SimpleTypeConstructor MULTI_POLYGON_TYPE_CONSTRUCTOR =
      (TypeBuilder typeBuilder, Symbol symbol) -> {
    return MULTI_POLYGON;
  };

  public static final TypeInformation MULTI_POLYGON_TYPE_INFORMATION = new TypeInformation(
      "multipolygon",
      MultiGeom.class,
      MultiPolygon.class,
      MULTI_POLYGON_TYPE_CONSTRUCTOR
  );

  private final Class<? extends GeometryCollection> internalType;

  @Getter
  private final Type containedType;

  @Getter
  private final Type underlyingType = Types.GEOMETRY;

  @Override
  public Object coerce(Object value) throws CoercionException {
    Object coerced = underlyingType.coerce(value);
    if (!internalType().isAssignableFrom(coerced.getClass())) {
      throw new CoercionException(value, this, "Can not coerce object of type %s", value.getClass());
    }
    return coerced;
  }

  @Override
  public Class<?> internalType() {
    return internalType;
  }

  @Override
  public String toString() {
    return String.format("Multi%s", containedType);
  }

  @Override
  public <T extends Type> boolean isA(Class<T> type) {
    return type.isAssignableFrom(getClass());
  }

  @Override
  public <T, U> U visit(TypeVisitor<T, U> tv, T data) {
    // we err on the side of being a containing type - the wrapping type is a bit of a kludge/convenience to support
    // this type being covariant with a plain ol' Geom - this way the visitor gets to see our geometry type next (which
    // in turn will eventually yield Geom, as they wrap that)
    return ContainingType.super.visit(tv, data);
  }

}
